/**
*  Class TEA - The Tiny Encryption Algorithm
*  <P>
*  Coded Mr. Tines &lt;tines@windsong.demon.co.uk&gt; 1998
*  <p>
*  The Tiny Encryption Algorithm of Wheeler and Needham
*  based on the code on the Vader web site at Bradford UK
*  Brought into the Angerona context and released into the public domain
*  <P>
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*  <P>
* @author Mr. Tines
* @version 1.0 23-Dec-1998
*/


package uk.co.demon.windsong.crypt.cea;

import uk.co.demon.windsong.crypt.cea.CEA;

public class TEA implements CEA
{
    /**
    * key size in bytes
    */
    private static final int TEAKEYSIZE  = 16;
    /**
    * block size in bytes
    */
    private static final int TEABLOCKSIZE = 8;
    /**
    *  delta is 1/phi * 2^32 where
    *    phi is the Golden ratio, 1.618...
    */
    private static final int DELTA =0x9E3779B9;
    /**
    * round count
    */
    private static final int NROUNDS = 32;

    /**
    * key schedule - really equals the key
    */
    class Keyschedule
    {
        /**
        * key as big-endian integers 
        */
	    int[] k = new int [TEAKEYSIZE/4];

        /**
        * Encipher two int in place
        * @param Xl first half of block
        * @param Xr second half of block
        */
        void TEA_encipher(int[] Xl, int[] Xr)
        {
	        int y=Xl[0], z=Xr[0], sum=0, delta=DELTA, n=NROUNDS;

	        while(n-- > 0)
            {
   	            sum += delta;
                y += (z<<4)+k[0] ^ z+sum ^ (z>>5)+k[1];
                z += (y<<4)+k[2] ^ y+sum ^ (y>>5)+k[3];
            }
            Xl[0] = y;
            Xr[0] = z;
        }

        /**
        * Decipher two int in place
        * @param Xl first half of block
        * @param Xr second half of block
        */
        void TEA_decipher(int[] Xl, int[] Xr)
        {
	        int y=Xl[0], z=Xr[0], sum=DELTA*NROUNDS, delta=DELTA, n=NROUNDS;

	        while(n-- > 0)
            {
                z -= (y<<4)+k[2] ^ y+sum ^ (y>>5)+k[3];
                y -= (z<<4)+k[0] ^ z+sum ^ (z>>5)+k[1];
   	            sum -= delta;
            }
            Xl[0] = y;
            Xr[0] = z;
        }

        /**
        * Pack key bytes into local ints
        * @param key byte array
        * @param offset start of key data
        */
        Keyschedule(byte[] key, int offset)
        {
            int i, j;  /* counters */

            /* Re-order the key bytes to local architecture */
            for (i=0, j=offset; i < TEAKEYSIZE/4; ++i, j+=4)
            {
		        k[i] =  ((key[j]&0xFF)<<24) |
                        ((key[j+1]&0xFF)<<16) |
                        ((key[j+2]&0xFF<<8)) |
                        (key[j+3]&0xFF);
            }
        }

        /**
        * Wipe key schedule information
        */
        void destroy()
        {
            if(k != null)
            {
                for(int i=0; i<k.length; ++i) k[i] = 0;
            }
            k = null;
        }
    }

    /**
    * Is the jacket doing triple encryption?
    */
    private boolean triple = false;
    /**
    * The key schedule data
    */
    private Keyschedule[] ks = null;

    /**
    * Initialise the object with one or three key blocks
    * @param key array of key bytes, 1 or 3 key block lengths
    * @param triple true if three keys for triple application
    */
    public void init(byte[] key, int offset, boolean triple)
    {
        this.triple = triple;
	    int keys = triple ? 3 : 1;
	    int i;

        ks = new Keyschedule[keys];
	    for(i=0; i < keys; i++)
	    {
            ks[i] = new Keyschedule(key, offset+i*TEAKEYSIZE);
	    }
    }

    /**
    * Transform one block in ecb mode
    * @param encrypt true if forwards transformation
    * @param in input block
    * @param offin offset into block of input data
    * @param out output block
    * @param offout offset into block of output data
    */
    public final void ecb(boolean encrypt, byte[] in, int offin, byte[] out, int offout)
    {
	    int keys = triple ? 3 : 1;
	    int[] text0 = new int[1];
	    int[] text1 = new int[1];
        int b = 0;
        if(triple & !encrypt) b=2;
        int delta = encrypt ? 1 : -1;

	    /* pack byte streams in MSB-first form into the 32bit integers */
	    /* so that the MSB becomes the high byte, regardless of architecture */
	    text0[0] = ((in[0+offin]&0xFF)<<24) |
                   ((in[1+offin]&0xFF)<<16) |
     		       ((in[2+offin]&0xFF)<<8)  |
                    (in[3+offin]&0xFF);
	    text1[0] = ((in[4+offin]&0xFF)<<24) |
                   ((in[5+offin]&0xFF)<<16) |
     		       ((in[6+offin]&0xFF)<<8)  |
                    (in[7+offin]&0xFF);

	    for(int i=0; i<keys; i++, b+=delta)
	    {
		    if(encrypt) ks[b].TEA_encipher(text0, text1);
		    else ks[b].TEA_decipher(text0, text1);
	    }

	    /* and unpack back into MSB-first format*/
	    out[0] = (byte)((text0[0]>>24) & 0xFF);
        out[1] = (byte)((text0[0]>>16) & 0xFF);
	    out[2] = (byte)((text0[0]>> 8) & 0xFF);
        out[3] = (byte)( text0[0] & 0xFF);
	    out[4] = (byte)((text1[0]>>24) & 0xFF);
        out[5] = (byte)((text1[0]>>16) & 0xFF);
	    out[6] = (byte)((text1[0]>> 8) & 0xFF);
        out[7] = (byte)( text1[0] & 0xFF);
    }


    /**
    * Wipe key schedule information
    */
    public final void destroy()
    {
        if(ks != null)
        {
            for(int i=0; i<ks.length; ++i)
            {
                ks[i].destroy();
                ks[i] = null;
            }
            ks = null;
        }
        triple = false;
    }

    /**
    * default and only constructor
    */
    public TEA()
    {
    }

    /**
    * Drives test harness
    */
    public static void naim(String[] args)
    {
        test();
    }

    /**
    * stub awaiting test vectors
    */
    public static void test()
    {
/*
	Enough code to run the above with a couple of known test vectors to ensure
	that 1) it is invertible and 2) it is TEA - when we have the test vectors!
*/
    }
    
    /**
    * Provide infomation of desired key size
    * @return byte length of key
    */
    public int getKeysize()
    {
        return TEAKEYSIZE;
    }

    /**
    * Provide infomation of algorithm block size
    * @return byte length of block
    */
    public int getBlocksize()    {
        return TEABLOCKSIZE;
    }
}
