/**
 * The HAVAL hashing function
 * Public domain implementation by Paulo S.L.M. Barreto &lt;pbarreto@uninet.com.br&gt;
 * Version 1.1 (1997.04.07)
 *<p>
 * Modified for MDA interface & Java port Mr. Tines &lt;tines@windsong.demon.co.uk&gt;
 * 26-Dec-1998
*  <p>
 * =============================================================================
 * <p>
 * Differences from version 1.0 (1997.04.03):
 * <p>
 * - Replaced function F5 by an optimized version (saving a boolean operation).
 *   Thanks to Wei Dai &lt;weidai@eskimo.com&gt; for this improvement.
 *  <p>
 * =============================================================================
 *  <p>
 * Reference: Zheng, Y., Pieprzyk, J., Seberry, J.:
 * "HAVAL - a one-way hashing algorithm with variable length of output",
 * Advances in Cryptology (AusCrypt'92), LNCS 718 (1993), 83-104, Springer-Verlag.
*  <P>
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*  <P>
* @author Mr. Tines
* @version 1.0 27-Dec-1998
*/

package uk.co.demon.windsong.crypt.mda;

public class Haval implements MDA
{

    /**
    * Initialise a HAVAL hashing context as per EBP
    */
    public Haval()
    {
        this(97);
    }

	/**
    * Initializes a HAVAL hashing context according to the desired
	* number of passes and hash length.
    * @param fixed if true use 0x01 as padding not 0x08
    * @param passes number of passes must be 3, 4, or 5.
	* @param length hash length must be 128, 160, 192, 224, or 256.
    */
    public Haval(boolean fixed, int passes, int length)
    {
        if(passes < 3) passes = 3;
        if(passes > 5) passes = 5;
        if(length != 128 && length !=160 && length != 192
           && length != 224) length = 256;

        ebp = !fixed;
        this.passes = passes;
        this.hashLength = length;
        init();
    }

    /**
    * Initialise a HAVAL hashing context as per EBP
    * @param type byte value 97+
    */
    public Haval(int type)
    {
        this((type > 0x80), getPasses(type), getSize(type));
    }



    /**
    * sets initial state and magic numbers
    */
    private void init()
    {
        int i;
        md[0] = 0x243F6A88;
        md[1] = 0x85A308D3;
        md[2] = 0x13198A2E;
        md[3] = 0x03707344;
        md[4] = 0xA4093822;
        md[5] = 0x299F31D0;
        md[6] = 0x082EFA98;
        md[7] = 0xEC4E6C89;

        for(i=0; i<block.length; ++i) block[i] = 0;
	    occupied = 0;
	    bitCount = 0;
        for(i=0; i<temp.length; ++i) temp[i] = 0;
    }


   /**
    * Feeds a batch of bytes into the hash
    * @param data the byte values
    * @param offset the first byte index to take
    * @param length the number of bytes to take
    */
    public void update(byte[] data, int offset, int length)
    {

	    if (data == null || length == 0)
        {
		    return; /* nothing to do */
	    }

	    /* update bit count: */
	    bitCount += length << 3;

	    /* if the data buffer is not enough to complete */
	    /* the context data block, just append it: */
	    if (occupied + length < 128)
        {
            System.arraycopy(data, offset, block, occupied, length);
            occupied += length;
            return; /* delay processing */
	    }

	    /* complete the context data block: */
        System.arraycopy(data, offset, block, occupied, 128-occupied);
	    offset += 128 - occupied;
	    length -= 128 - occupied;

	    switch (passes)
        {
	        case 3:
		    /* process the completed context data block: */
		    transform3 (block, 0);
		    /* process data in chunks of 128 bytes: */
		    while (length >= 128)
            {
			    transform3 (data, offset);
			    offset += 128;
			    length -= 128;
		    }
		    break;
	        case 4:
		    /* process the completed context data block: */
		    transform4 (block, 0);
		    /* process data in chunks of 128 bytes: */
		    while (length >= 128)
            {
			    transform4 (data, offset);
			    offset += 128;
			    length -= 128;
		    }
		    break;
	        case 5:
		    /* process the completed context data block: */
		    transform5 (block, 0);
		    /* process data in chunks of 128 bytes: */
		    while (length >= 128)
            {
			    transform5 (data, offset);
			    offset += 128;
			    length -= 128;
		    }
		    break;
	    }

        System.arraycopy(data, offset, block, 0, length);
        occupied = length;
    }

   /**
    * Feeds a  byte into the hash
    * @param data the byte value
    */
    public void update(byte data)
    {
        byte[] temp = {data};
        update(temp, 0, temp.length);
    }

    public byte[] digest()
    {
	    /* append toggle to the context data block: */
        if(ebp)
        {
   	        block[occupied] = (byte)0x80;  /* EBP gets this wrong */
	    }
        else
        {
		    block[occupied] = (byte) 0x01; /* corrected from 0x80 */
        }

	    /* pad the message with null bytes to make it 944 (mod 1024) bits long: */
	    if (occupied++ >= 118)
        {
		    /* no room for tail data on the current context block */
            for(int i=occupied; i<128; ++i) block[i]=0;

		    /* process the completed context data block: */
		    switch (passes)
            {
		        case 3:
			    transform3 (block, 0);
			    break;
		        case 4:
			    transform4 (block, 0);
			    break;
		        case 5:
			    transform5 (block, 0);
			    break;
		    }
            for(int j=0; j<118; ++j) block[j] = 0;
	    }
        else
        {
            for(int k=occupied; k<118;++k) block[k] = 0;
	    }
	    /* append tail data and process last (padded) message block: */
        if(ebp) /* EBP got this wrong too!, hardcoding it at the max */
        {
		    block[118] =  (byte) (
			((256 & 0x03) << 6) |
			((5     & 0x07) << 3) |
			(HAVAL_VERSION           & 0x07));
		    block[119] = (byte)(256 >>> 2);
        }
        else
        {
		    block[118] =  (byte) (
			((hashLength & 0x03) << 6) |
			((passes     & 0x07) << 3) |
			(HAVAL_VERSION           & 0x07));
		    block[119] = (byte)(hashLength >>> 2);
        }
        for(int ix=0; ix<8;++ix)
        {
            block[120+ix] = (byte)(0xFF&(bitCount>>(8*ix)));
        }

	    switch (passes)
        {
	        case 3:
		    transform3 (block, 0);
		    break;
	        case 4:
		    transform4 (block, 0);
		    break;
	        case 5:
		    transform5 (block, 0);
		    break;
	    }

	    /* fold 256-bit digest to fit the desired hash length (blaargh!): */
	    switch (hashLength)
        {
	        case 128:
		    md[3] +=
			( (md[7] & 0xFF000000)
			| (md[6] & 0x00FF0000)
			| (md[5] & 0x0000FF00)
			| (md[4] & 0x000000FF)
			);
		    md[2] +=
			(((md[7] & 0x00FF0000)
			| (md[6] & 0x0000FF00)
			| (md[5] & 0x000000FF)
			) << 8) |
			( (md[4] & 0xFF000000) >>> 24);
		    md[1] +=
			(((md[7] & 0x0000FF00)
			| (md[6] & 0x000000FF)) << 16) |
			(((md[5] & 0xFF000000)
			| (md[4] & 0x00FF0000)) >>> 16);
		    md[0] +=
			(((md[6] & 0xFF000000)
			| (md[5] & 0x00FF0000)
			| (md[4] & 0x0000FF00)
			) >>> 8) |
			( (md[7] & 0x000000FF) << 24);
		    break;
	        case 160:
		    md[4] +=
			((md[7] & 0xFE000000) |   (md[6] & 0x01F80000) |
            (md[5] & 0x0007F000)) >>> 12;
		    md[3] +=
			((md[7] & 0x01F80000) |   (md[6] & 0x0007F000) |
            (md[5] & 0x00000FC0)) >>> 6;
		    md[2] +=
			((md[7] & 0x0007F000) |   (md[6] & 0x00000FC0) |
            (md[5] & 0x0000003F));
		    md[1] +=
			ROTR(((md[7] & 0x00000FC0) | (md[6] & 0x0000003F) |
            (md[5] & 0xFE000000)), 25);
		    md[0] +=
			ROTR((md[7] & 0x0000003F) | (md[6] & 0xFE000000) |
            (md[5] & 0x01F80000), 19);
		    break;
	        case 192:
		    md[5] +=
			    ((md[7] & 0xFC000000) | (md[6] & 0x03E00000)) >>> 21;
		    md[4] +=
			    ((md[7] & 0x03E00000) | (md[6] & 0x001F0000)) >>> 16;
		    md[3] +=
			    ((md[7] & 0x001F0000) | (md[6] & 0x0000FC00)) >>> 10;
		    md[2] +=
			    ((md[7] & 0x0000FC00) | (md[6] & 0x000003E0)) >>>  5;
		    md[1] +=
			    ((md[7] & 0x000003E0) | (md[6] & 0x0000001F));
		    md[0] +=
			    ROTR ((md[7] & 0x0000001F) | (md[6] & 0xFC000000), 26);
		    break;
	        case 224:
		    md[6] += (md[7]      ) & 0x0000000F;
		    md[5] += (md[7] >>>  4) & 0x0000001F;
		    md[4] += (md[7] >>>  9) & 0x0000000F;
		    md[3] += (md[7] >>> 13) & 0x0000001F;
		    md[2] += (md[7] >>> 18) & 0x0000000F;
		    md[1] += (md[7] >>> 22) & 0x0000001F;
		    md[0] += (md[7] >>> 27) & 0x0000001F;
		    break;
	        case 256:
		    break;
	    }

        byte[] output = new byte[hashLength/8];
        flip32(output);
        init();
        return output;
    }

    /**
    * flips each 32-bit section from md array to output
    * @param output byte array with hash in
    */
    private final void flip32(byte[] output)
    {
	    int count = output.length/4;
        for(int i=0; i<count;++i)
        {
            output[4*i+3] = (byte)((md[i]>>>24)&0xFF);
            output[4*i+2] = (byte)((md[i]>>>16)&0xFF);
            output[4*i+1] = (byte)((md[i]>>> 8)&0xFF);
            output[4*i+0] = (byte)((md[i])&0xFF);
        }
    }

    /**
    * HAVAL parameters */
    private int passes, hashLength;
    /**
    * message digest (fingerprint)
    */
	private int[] md = new int[8];
    /**
    * context data block
    */
	private byte[] block = new byte [128];
    /**
    * number of occupied bytes in the data block
    */
	private int occupied;
    /**
    * 64-bit message bit count
    */
	long bitCount;
    /**
    * temporary buffer
    */
	int[] temp = new int[8];
    /**
    * controls padding style
    */
    boolean  ebp;

    /**
    * magic number
    */
    private static final int HAVAL_VERSION = 1;

    /**
    * round functions
    */
    private static final int F1(int X6, int X5, int X4, int X3, int X2, int X1, int X0)
    {
	    return ( ((X1) & ((X4) ^ (X0))) ^ ((X2) & (X5)) ^ ((X3) & (X6)) ^ (X0));
    }
    private static final int F2(int X6, int X5, int X4, int X3, int X2, int X1, int X0)
    {
        return 	( ((X2) & ( ((X1) & (~(X3))) ^ ((X4) & (X5)) ^ (X6) ^ (X0))) ^
	    (((X4) & ((X1) ^ (X5))) ^ ((X3) & (X5)) ^ (X0)));
    }

    private static final int F3(int X6, int X5, int X4, int X3, int X2, int X1, int X0)
    {
        return 	( ((X3) & (((X1) & (X2)) ^ (X6) ^ (X0))) ^ ((X1) & (X4)) ^ ((X2) & (X5)) ^ (X0));
    }

    private static final int F4(int X6, int X5, int X4, int X3, int X2, int X1, int X0)
    {
        return 	( ((X4) & ( ((~(X2)) & (X5)) ^ ((X3) | (X6)) ^ (X1) ^ (X0) )) ^
    	((X3) & (((X1) & (X2)) ^ (X5) ^ (X6))) ^ ((X2) & (X6)) ^ (X0) );
    }

    private static final int F5(int X6, int X5, int X4, int X3, int X2, int X1, int X0)
    {
	    return (  (  (X1) & ( (X4) ^ ((X0) & (X2) & (X3)) )   ) ^ (((X2) ^ (X0)) & (X5)) ^ ((X3) & (X6)) ^ (X0)  );
    }

    private static final int ROTR(int v, int n)
    {
        return (v >>> n) | (v << (32-n));
    }

    /**
    * decode number of passes from algorithm byte value
    * @param ebpType byte value 97-111 as int
    */
    private static final int getPasses(int ebpType)
    {
	    int p;
        if(ebpType > 128) { ebpType = 97 + (ebpType - 133); }
	    if(ebpType < 97) return 5;
        else if(ebpType <= 111) p = (ebpType - 111);
        else return 5;
        return 5 -(p/5);
    }

    /**
    * decode length of hash from algorithm byte value
    * @param ebpType byte value 97-111 as int
    */
    private static final int getSize(int ebpType)
    {
	    int p;
        if(ebpType > 128) { ebpType = 97 + (ebpType - 133); }
	    if(ebpType < 97) return 256;
        else if(ebpType <= 111) p = (ebpType - 111)%5;
        else return 256;

        switch (p)
        {
   	        case 0: return 256;
            case 1: return 224;
            case 2: return 192;
            case 3: return 160;
            case 4: return 128;
        }
        return 256;
    }

    /**
    * drives test harness
    */
    public static void main(String[] args)
    {
        test();
    }

    /**
    * pretty printer
    * @param tag string tag
    * @param digest 20 byte output from the hash
    */
    private static final void printDigest(String tag, byte[] digest)
    {
	    int i,j;
        System.out.print(tag+" ");
	    for (i = 0; i < digest.length/4; i++)
        {
		    for (j = 0; j < 4; j++)
            {
                if((0xFF&digest[i*4+j]) < 0x10)
                    System.out.print("0"+Integer.toHexString(0xFF&digest[i*4+j]));
                else
                    System.out.print(Integer.toHexString(0xFF&digest[i*4+j]));
		    }
            System.out.print(" ");
	    }
        System.out.println("");
    }

    /**
    * Updates, treating a string as a stream of its low bytes only -
    * simplifies test vectors
    * @param s String
    */
    private final void ascii(String s)
    {
        byte[] temp = new byte[s.length()];
        for(int i=0; i<temp.length; ++i) temp[i] = (byte)s.charAt(i);
        update(temp, 0, temp.length);
    }

    /**
    * Runs test harness
    * @param p number of passes
    */
    private static final void test()
    {
        Haval hc = new Haval();
        hc.ebp = false;

        hc.passes = 3;
        hc.hashLength = 128;
		hc.ascii("");
		System.out.println("HAVAL(3,128,\"\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = C68F3991 3F901F3D DF44C707 357A7D70");

        hc.hashLength = 160;
		hc.ascii("a");
		System.out.println("HAVAL(3,160,\"a\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 4DA08F51 4A7275DB C4CECE4A 34738598 3983A830");

        hc.hashLength = 192;
		hc.ascii("HAVAL");
		System.out.println("HAVAL(3,192,\"HAVAL\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 8DA26DDA B4317B39 2B22B638 998FE65B 0FBE4610 D345CF89");

        hc.hashLength = 224;
		hc.ascii("0123456789");
		System.out.println("HAVAL(3,224,\"0123456789\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = EE345C97 A58190BF 0F38BF7C E890231A A5FCF986 2BF8E7BE BBF76789");

        hc.hashLength = 256;
		hc.ascii("abcdefghijklmnopqrstuvwxyz");
		System.out.println("HAVAL(3,256,\"abcdefghijklmnopqrstuvwxyz\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 72FAD4BD E1DA8C83 32FB6056 1A780E7F 504F2154 7B986868 24FC33FC 796AFA76");

        hc.ascii("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
		System.out.println("HAVAL(3,256,\"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 899397D9 6489281E 9E76D5E6 5ABAB751 F312E06C 06C07C9C 1D42ABD3 1BB6A404");

        hc.passes = 3;
        hc.hashLength = 128;
		hc.ascii("");
		System.out.println("HAVAL(4,128,\"\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = EE6BBF4D 6A46A679 B3A856C8 8538BB98");

        hc.hashLength = 160;
		hc.ascii("a");
		System.out.println("HAVAL(4,160,\"a\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = E0A5BE29 62733203 4D4DD8A9 10A1A0E6 FE04084D");

        hc.hashLength = 192;
		hc.ascii("HAVAL");
		System.out.println("HAVAL(4,192,\"HAVAL\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 0C1396D7 772689C4 6773F3DA ACA4EFA9 82ADBFB2 F1467EEA");

        hc.hashLength = 224;
		hc.ascii("0123456789");
		System.out.println("HAVAL(4,224,\"0123456789\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = BEBD7816 F09BAEEC F8903B1B 9BC672D9 FA428E46 2BA699F8 14841529");

        hc.hashLength = 256;
		hc.ascii("abcdefghijklmnopqrstuvwxyz");
		System.out.println("HAVAL(4,256,\"abcdefghijklmnopqrstuvwxyz\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 124F6EB6 45DC4076 37F8F719 CC312500 89C89903 BF1DB8FA C21EA461 4DF4E99A");

		hc.ascii("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
		System.out.println("HAVAL(4,256,\"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 46A3A1DF E867EDE6 52425CCD 7FE80065 37EAD263 72251686 BEA286DA 152DC35A");

        hc.passes = 5;
        hc.hashLength = 128;
		hc.ascii("");
		System.out.println("HAVAL(5,128,\"\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 184B8482 A0C050DC A54B59C7 F05BF5DD");

        hc.hashLength = 160;
		hc.ascii("a");
		System.out.println("HAVAL(5,160,\"a\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = F5147DF7 ABC5E3C8 1B031268 927C2B57 61B5A2B5");

        hc.hashLength = 192;
		hc.ascii("HAVAL");
		System.out.println("HAVAL(5,192,\"HAVAL\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 794A896D 1780B76E 2767CC40 11BAD888 5D5CE6BD 835A71B8");

        hc.hashLength = 224;
		hc.ascii("0123456789");
		System.out.println("HAVAL(5,224,\"0123456789\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = 59836D19 269135BC 815F37B2 AEB15F89 4B5435F2 C698D577 16760F2B");

        hc.hashLength = 256;
		hc.ascii("abcdefghijklmnopqrstuvwxyz");
		System.out.println("HAVAL(5,256,\"abcdefghijklmnopqrstuvwxyz\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = C9C7D8AF A159FD9E 965CB83F F5EE6F58 AEDA352C 0EFF0055 48153A61 551C38EE");

		hc.ascii("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
		System.out.println("HAVAL(5,256,\"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789\")");
		printDigest ("evaluated", hc.digest());
		System.out.println("expected  = B45CB6E6 2F2B1320 E4F8F1B0 B273D45A DD47C321 FD23999D CF403AC3 7636D963");
    }

    void transform3 (byte[] D, int offset)
    {
        int[] W = new int[32];
        int i;
        for(i=0; i<32; ++i)
        {
            W[i] =
                ((D[offset+3+4*i]&0xFF)<<24) |
                ((D[offset+2+4*i]&0xFF)<<16) |
                ((D[offset+1+4*i]&0xFF)<< 8) |
                ((D[offset+0+4*i]&0xFF)    );
        }


	/* PASS 1: */
	temp[7] = ROTR (F1 (md[1], md[0], md[3], md[5], md[6], md[2], md[4]), 7) + ROTR (md[7], 11) + W[ 0];
	temp[6] = ROTR (F1 (md[0], temp[7], md[2], md[4], md[5], md[1], md[3]), 7) + ROTR (md[6], 11) + W[ 1];
	temp[5] = ROTR (F1 (temp[7], temp[6], md[1], md[3], md[4], md[0], md[2]), 7) + ROTR (md[5], 11) + W[ 2];
	temp[4] = ROTR (F1 (temp[6], temp[5], md[0], md[2], md[3], temp[7], md[1]), 7) + ROTR (md[4], 11) + W[ 3];
	temp[3] = ROTR (F1 (temp[5], temp[4], temp[7], md[1], md[2], temp[6], md[0]), 7) + ROTR (md[3], 11) + W[ 4];
	temp[2] = ROTR (F1 (temp[4], temp[3], temp[6], md[0], md[1], temp[5], temp[7]), 7) + ROTR (md[2], 11) + W[ 5];
	temp[1] = ROTR (F1 (temp[3], temp[2], temp[5], temp[7], md[0], temp[4], temp[6]), 7) + ROTR (md[1], 11) + W[ 6];
	temp[0] = ROTR (F1 (temp[2], temp[1], temp[4], temp[6], temp[7], temp[3], temp[5]), 7) + ROTR (md[0], 11) + W[ 7];

	temp[7] = ROTR (F1 (temp[1], temp[0], temp[3], temp[5], temp[6], temp[2], temp[4]), 7) + ROTR (temp[7], 11) + W[ 8];
	temp[6] = ROTR (F1 (temp[0], temp[7], temp[2], temp[4], temp[5], temp[1], temp[3]), 7) + ROTR (temp[6], 11) + W[ 9];
	temp[5] = ROTR (F1 (temp[7], temp[6], temp[1], temp[3], temp[4], temp[0], temp[2]), 7) + ROTR (temp[5], 11) + W[10];
	temp[4] = ROTR (F1 (temp[6], temp[5], temp[0], temp[2], temp[3], temp[7], temp[1]), 7) + ROTR (temp[4], 11) + W[11];
	temp[3] = ROTR (F1 (temp[5], temp[4], temp[7], temp[1], temp[2], temp[6], temp[0]), 7) + ROTR (temp[3], 11) + W[12];
	temp[2] = ROTR (F1 (temp[4], temp[3], temp[6], temp[0], temp[1], temp[5], temp[7]), 7) + ROTR (temp[2], 11) + W[13];
	temp[1] = ROTR (F1 (temp[3], temp[2], temp[5], temp[7], temp[0], temp[4], temp[6]), 7) + ROTR (temp[1], 11) + W[14];
	temp[0] = ROTR (F1 (temp[2], temp[1], temp[4], temp[6], temp[7], temp[3], temp[5]), 7) + ROTR (temp[0], 11) + W[15];

	temp[7] = ROTR (F1 (temp[1], temp[0], temp[3], temp[5], temp[6], temp[2], temp[4]), 7) + ROTR (temp[7], 11) + W[16];
	temp[6] = ROTR (F1 (temp[0], temp[7], temp[2], temp[4], temp[5], temp[1], temp[3]), 7) + ROTR (temp[6], 11) + W[17];
	temp[5] = ROTR (F1 (temp[7], temp[6], temp[1], temp[3], temp[4], temp[0], temp[2]), 7) + ROTR (temp[5], 11) + W[18];
	temp[4] = ROTR (F1 (temp[6], temp[5], temp[0], temp[2], temp[3], temp[7], temp[1]), 7) + ROTR (temp[4], 11) + W[19];
	temp[3] = ROTR (F1 (temp[5], temp[4], temp[7], temp[1], temp[2], temp[6], temp[0]), 7) + ROTR (temp[3], 11) + W[20];
	temp[2] = ROTR (F1 (temp[4], temp[3], temp[6], temp[0], temp[1], temp[5], temp[7]), 7) + ROTR (temp[2], 11) + W[21];
	temp[1] = ROTR (F1 (temp[3], temp[2], temp[5], temp[7], temp[0], temp[4], temp[6]), 7) + ROTR (temp[1], 11) + W[22];
	temp[0] = ROTR (F1 (temp[2], temp[1], temp[4], temp[6], temp[7], temp[3], temp[5]), 7) + ROTR (temp[0], 11) + W[23];

	temp[7] = ROTR (F1 (temp[1], temp[0], temp[3], temp[5], temp[6], temp[2], temp[4]), 7) + ROTR (temp[7], 11) + W[24];
	temp[6] = ROTR (F1 (temp[0], temp[7], temp[2], temp[4], temp[5], temp[1], temp[3]), 7) + ROTR (temp[6], 11) + W[25];
	temp[5] = ROTR (F1 (temp[7], temp[6], temp[1], temp[3], temp[4], temp[0], temp[2]), 7) + ROTR (temp[5], 11) + W[26];
	temp[4] = ROTR (F1 (temp[6], temp[5], temp[0], temp[2], temp[3], temp[7], temp[1]), 7) + ROTR (temp[4], 11) + W[27];
	temp[3] = ROTR (F1 (temp[5], temp[4], temp[7], temp[1], temp[2], temp[6], temp[0]), 7) + ROTR (temp[3], 11) + W[28];
	temp[2] = ROTR (F1 (temp[4], temp[3], temp[6], temp[0], temp[1], temp[5], temp[7]), 7) + ROTR (temp[2], 11) + W[29];
	temp[1] = ROTR (F1 (temp[3], temp[2], temp[5], temp[7], temp[0], temp[4], temp[6]), 7) + ROTR (temp[1], 11) + W[30];
	temp[0] = ROTR (F1 (temp[2], temp[1], temp[4], temp[6], temp[7], temp[3], temp[5]), 7) + ROTR (temp[0], 11) + W[31];

	/* PASS 2: */

	temp[7] = ROTR (F2 (temp[4], temp[2], temp[1], temp[0], temp[5], temp[3], temp[6]), 7) + ROTR (temp[7], 11) + W[ 5] + 0x452821E6;
	temp[6] = ROTR (F2 (temp[3], temp[1], temp[0], temp[7], temp[4], temp[2], temp[5]), 7) + ROTR (temp[6], 11) + W[14] + 0x38D01377;
	temp[5] = ROTR (F2 (temp[2], temp[0], temp[7], temp[6], temp[3], temp[1], temp[4]), 7) + ROTR (temp[5], 11) + W[26] + 0xBE5466CF;
	temp[4] = ROTR (F2 (temp[1], temp[7], temp[6], temp[5], temp[2], temp[0], temp[3]), 7) + ROTR (temp[4], 11) + W[18] + 0x34E90C6C;
	temp[3] = ROTR (F2 (temp[0], temp[6], temp[5], temp[4], temp[1], temp[7], temp[2]), 7) + ROTR (temp[3], 11) + W[11] + 0xC0AC29B7;
	temp[2] = ROTR (F2 (temp[7], temp[5], temp[4], temp[3], temp[0], temp[6], temp[1]), 7) + ROTR (temp[2], 11) + W[28] + 0xC97C50DD;
	temp[1] = ROTR (F2 (temp[6], temp[4], temp[3], temp[2], temp[7], temp[5], temp[0]), 7) + ROTR (temp[1], 11) + W[ 7] + 0x3F84D5B5;
	temp[0] = ROTR (F2 (temp[5], temp[3], temp[2], temp[1], temp[6], temp[4], temp[7]), 7) + ROTR (temp[0], 11) + W[16] + 0xB5470917;

	temp[7] = ROTR (F2 (temp[4], temp[2], temp[1], temp[0], temp[5], temp[3], temp[6]), 7) + ROTR (temp[7], 11) + W[ 0] + 0x9216D5D9;
	temp[6] = ROTR (F2 (temp[3], temp[1], temp[0], temp[7], temp[4], temp[2], temp[5]), 7) + ROTR (temp[6], 11) + W[23] + 0x8979FB1B;
	temp[5] = ROTR (F2 (temp[2], temp[0], temp[7], temp[6], temp[3], temp[1], temp[4]), 7) + ROTR (temp[5], 11) + W[20] + 0xD1310BA6;
	temp[4] = ROTR (F2 (temp[1], temp[7], temp[6], temp[5], temp[2], temp[0], temp[3]), 7) + ROTR (temp[4], 11) + W[22] + 0x98DFB5AC;
	temp[3] = ROTR (F2 (temp[0], temp[6], temp[5], temp[4], temp[1], temp[7], temp[2]), 7) + ROTR (temp[3], 11) + W[ 1] + 0x2FFD72DB;
	temp[2] = ROTR (F2 (temp[7], temp[5], temp[4], temp[3], temp[0], temp[6], temp[1]), 7) + ROTR (temp[2], 11) + W[10] + 0xD01ADFB7;
	temp[1] = ROTR (F2 (temp[6], temp[4], temp[3], temp[2], temp[7], temp[5], temp[0]), 7) + ROTR (temp[1], 11) + W[ 4] + 0xB8E1AFED;
	temp[0] = ROTR (F2 (temp[5], temp[3], temp[2], temp[1], temp[6], temp[4], temp[7]), 7) + ROTR (temp[0], 11) + W[ 8] + 0x6A267E96;

	temp[7] = ROTR (F2 (temp[4], temp[2], temp[1], temp[0], temp[5], temp[3], temp[6]), 7) + ROTR (temp[7], 11) + W[30] + 0xBA7C9045;
	temp[6] = ROTR (F2 (temp[3], temp[1], temp[0], temp[7], temp[4], temp[2], temp[5]), 7) + ROTR (temp[6], 11) + W[ 3] + 0xF12C7F99;
	temp[5] = ROTR (F2 (temp[2], temp[0], temp[7], temp[6], temp[3], temp[1], temp[4]), 7) + ROTR (temp[5], 11) + W[21] + 0x24A19947;
	temp[4] = ROTR (F2 (temp[1], temp[7], temp[6], temp[5], temp[2], temp[0], temp[3]), 7) + ROTR (temp[4], 11) + W[ 9] + 0xB3916CF7;
	temp[3] = ROTR (F2 (temp[0], temp[6], temp[5], temp[4], temp[1], temp[7], temp[2]), 7) + ROTR (temp[3], 11) + W[17] + 0x0801F2E2;
	temp[2] = ROTR (F2 (temp[7], temp[5], temp[4], temp[3], temp[0], temp[6], temp[1]), 7) + ROTR (temp[2], 11) + W[24] + 0x858EFC16;
	temp[1] = ROTR (F2 (temp[6], temp[4], temp[3], temp[2], temp[7], temp[5], temp[0]), 7) + ROTR (temp[1], 11) + W[29] + 0x636920D8;
	temp[0] = ROTR (F2 (temp[5], temp[3], temp[2], temp[1], temp[6], temp[4], temp[7]), 7) + ROTR (temp[0], 11) + W[ 6] + 0x71574E69;

	temp[7] = ROTR (F2 (temp[4], temp[2], temp[1], temp[0], temp[5], temp[3], temp[6]), 7) + ROTR (temp[7], 11) + W[19] + 0xA458FEA3;
	temp[6] = ROTR (F2 (temp[3], temp[1], temp[0], temp[7], temp[4], temp[2], temp[5]), 7) + ROTR (temp[6], 11) + W[12] + 0xF4933D7E;
	temp[5] = ROTR (F2 (temp[2], temp[0], temp[7], temp[6], temp[3], temp[1], temp[4]), 7) + ROTR (temp[5], 11) + W[15] + 0x0D95748F;
	temp[4] = ROTR (F2 (temp[1], temp[7], temp[6], temp[5], temp[2], temp[0], temp[3]), 7) + ROTR (temp[4], 11) + W[13] + 0x728EB658;
	temp[3] = ROTR (F2 (temp[0], temp[6], temp[5], temp[4], temp[1], temp[7], temp[2]), 7) + ROTR (temp[3], 11) + W[ 2] + 0x718BCD58;
	temp[2] = ROTR (F2 (temp[7], temp[5], temp[4], temp[3], temp[0], temp[6], temp[1]), 7) + ROTR (temp[2], 11) + W[25] + 0x82154AEE;
	temp[1] = ROTR (F2 (temp[6], temp[4], temp[3], temp[2], temp[7], temp[5], temp[0]), 7) + ROTR (temp[1], 11) + W[31] + 0x7B54A41D;
	temp[0] = ROTR (F2 (temp[5], temp[3], temp[2], temp[1], temp[6], temp[4], temp[7]), 7) + ROTR (temp[0], 11) + W[27] + 0xC25A59B5;
    for(i=0; i<8;++i) System.out.print(" "+Integer.toHexString(temp[i]));
    System.out.println("");

	/* PASS 3: */

	temp[7] = ROTR (F3 (temp[6], temp[1], temp[2], temp[3], temp[4], temp[5], temp[0]), 7) + ROTR (temp[7], 11) + W[19] + 0x9C30D539;
	temp[6] = ROTR (F3 (temp[5], temp[0], temp[1], temp[2], temp[3], temp[4], temp[7]), 7) + ROTR (temp[6], 11) + W[ 9] + 0x2AF26013;
	temp[5] = ROTR (F3 (temp[4], temp[7], temp[0], temp[1], temp[2], temp[3], temp[6]), 7) + ROTR (temp[5], 11) + W[ 4] + 0xC5D1B023;
	temp[4] = ROTR (F3 (temp[3], temp[6], temp[7], temp[0], temp[1], temp[2], temp[5]), 7) + ROTR (temp[4], 11) + W[20] + 0x286085F0;
	temp[3] = ROTR (F3 (temp[2], temp[5], temp[6], temp[7], temp[0], temp[1], temp[4]), 7) + ROTR (temp[3], 11) + W[28] + 0xCA417918;
	temp[2] = ROTR (F3 (temp[1], temp[4], temp[5], temp[6], temp[7], temp[0], temp[3]), 7) + ROTR (temp[2], 11) + W[17] + 0xB8DB38EF;
	temp[1] = ROTR (F3 (temp[0], temp[3], temp[4], temp[5], temp[6], temp[7], temp[2]), 7) + ROTR (temp[1], 11) + W[ 8] + 0x8E79DCB0;
	temp[0] = ROTR (F3 (temp[7], temp[2], temp[3], temp[4], temp[5], temp[6], temp[1]), 7) + ROTR (temp[0], 11) + W[22] + 0x603A180E;

	temp[7] = ROTR (F3 (temp[6], temp[1], temp[2], temp[3], temp[4], temp[5], temp[0]), 7) + ROTR (temp[7], 11) + W[29] + 0x6C9E0E8B;
	temp[6] = ROTR (F3 (temp[5], temp[0], temp[1], temp[2], temp[3], temp[4], temp[7]), 7) + ROTR (temp[6], 11) + W[14] + 0xB01E8A3E;
	temp[5] = ROTR (F3 (temp[4], temp[7], temp[0], temp[1], temp[2], temp[3], temp[6]), 7) + ROTR (temp[5], 11) + W[25] + 0xD71577C1;
	temp[4] = ROTR (F3 (temp[3], temp[6], temp[7], temp[0], temp[1], temp[2], temp[5]), 7) + ROTR (temp[4], 11) + W[12] + 0xBD314B27;
	temp[3] = ROTR (F3 (temp[2], temp[5], temp[6], temp[7], temp[0], temp[1], temp[4]), 7) + ROTR (temp[3], 11) + W[24] + 0x78AF2FDA;
	temp[2] = ROTR (F3 (temp[1], temp[4], temp[5], temp[6], temp[7], temp[0], temp[3]), 7) + ROTR (temp[2], 11) + W[30] + 0x55605C60;
	temp[1] = ROTR (F3 (temp[0], temp[3], temp[4], temp[5], temp[6], temp[7], temp[2]), 7) + ROTR (temp[1], 11) + W[16] + 0xE65525F3;
	temp[0] = ROTR (F3 (temp[7], temp[2], temp[3], temp[4], temp[5], temp[6], temp[1]), 7) + ROTR (temp[0], 11) + W[26] + 0xAA55AB94;

	temp[7] = ROTR (F3 (temp[6], temp[1], temp[2], temp[3], temp[4], temp[5], temp[0]), 7) + ROTR (temp[7], 11) + W[31] + 0x57489862;
	temp[6] = ROTR (F3 (temp[5], temp[0], temp[1], temp[2], temp[3], temp[4], temp[7]), 7) + ROTR (temp[6], 11) + W[15] + 0x63E81440;
	temp[5] = ROTR (F3 (temp[4], temp[7], temp[0], temp[1], temp[2], temp[3], temp[6]), 7) + ROTR (temp[5], 11) + W[ 7] + 0x55CA396A;
	temp[4] = ROTR (F3 (temp[3], temp[6], temp[7], temp[0], temp[1], temp[2], temp[5]), 7) + ROTR (temp[4], 11) + W[ 3] + 0x2AAB10B6;
	temp[3] = ROTR (F3 (temp[2], temp[5], temp[6], temp[7], temp[0], temp[1], temp[4]), 7) + ROTR (temp[3], 11) + W[ 1] + 0xB4CC5C34;
	temp[2] = ROTR (F3 (temp[1], temp[4], temp[5], temp[6], temp[7], temp[0], temp[3]), 7) + ROTR (temp[2], 11) + W[ 0] + 0x1141E8CE;
	temp[1] = ROTR (F3 (temp[0], temp[3], temp[4], temp[5], temp[6], temp[7], temp[2]), 7) + ROTR (temp[1], 11) + W[18] + 0xA15486AF;
	temp[0] = ROTR (F3 (temp[7], temp[2], temp[3], temp[4], temp[5], temp[6], temp[1]), 7) + ROTR (temp[0], 11) + W[27] + 0x7C72E993;

	md[7] += temp[7] = ROTR (F3 (temp[6], temp[1], temp[2], temp[3], temp[4], temp[5], temp[0]), 7) + ROTR (temp[7], 11) + W[13] + 0xB3EE1411;
	md[6] += temp[6] = ROTR (F3 (temp[5], temp[0], temp[1], temp[2], temp[3], temp[4], temp[7]), 7) + ROTR (temp[6], 11) + W[ 6] + 0x636FBC2A;
	md[5] += temp[5] = ROTR (F3 (temp[4], temp[7], temp[0], temp[1], temp[2], temp[3], temp[6]), 7) + ROTR (temp[5], 11) + W[21] + 0x2BA9C55D;
	md[4] += temp[4] = ROTR (F3 (temp[3], temp[6], temp[7], temp[0], temp[1], temp[2], temp[5]), 7) + ROTR (temp[4], 11) + W[10] + 0x741831F6;
	md[3] += temp[3] = ROTR (F3 (temp[2], temp[5], temp[6], temp[7], temp[0], temp[1], temp[4]), 7) + ROTR (temp[3], 11) + W[23] + 0xCE5C3E16;
	md[2] += temp[2] = ROTR (F3 (temp[1], temp[4], temp[5], temp[6], temp[7], temp[0], temp[3]), 7) + ROTR (temp[2], 11) + W[11] + 0x9B87931E;
	md[1] += temp[1] = ROTR (F3 (temp[0], temp[3], temp[4], temp[5], temp[6], temp[7], temp[2]), 7) + ROTR (temp[1], 11) + W[ 5] + 0xAFD6BA33;
	md[0] += temp[0] = ROTR (F3 (temp[7], temp[2], temp[3], temp[4], temp[5], temp[6], temp[1]), 7) + ROTR (temp[0], 11) + W[ 2] + 0x6C24CF5C;

        for(i=0; i<32; ++i)W[i] = 0;
    } /* havalTransform3 */

    void transform4 (byte[] D, int offset)
    {
        int[] W = new int[32];
        int i;
        for(i=0; i<32; ++i)
        {
            W[i] =
                ((D[offset+3+4*i]&0xFF)<<24) |
                ((D[offset+2+4*i]&0xFF)<<16) |
                ((D[offset+1+4*i]&0xFF)<< 8) |
                ((D[offset+0+4*i]&0xFF)    );
        }

	/* PASS 1: */

	temp[7] = ROTR (F1 (md[2], md[6], md[1], md[4], md[5], md[3], md[0]), 7) + ROTR (md[7], 11) + W[ 0];
	temp[6] = ROTR (F1 (md[1], md[5], md[0], md[3], md[4], md[2], temp[7]), 7) + ROTR (md[6], 11) + W[ 1];
	temp[5] = ROTR (F1 (md[0], md[4], temp[7], md[2], md[3], md[1], temp[6]), 7) + ROTR (md[5], 11) + W[ 2];
	temp[4] = ROTR (F1 (temp[7], md[3], temp[6], md[1], md[2], md[0], temp[5]), 7) + ROTR (md[4], 11) + W[ 3];
	temp[3] = ROTR (F1 (temp[6], md[2], temp[5], md[0], md[1], temp[7], temp[4]), 7) + ROTR (md[3], 11) + W[ 4];
	temp[2] = ROTR (F1 (temp[5], md[1], temp[4], temp[7], md[0], temp[6], temp[3]), 7) + ROTR (md[2], 11) + W[ 5];
	temp[1] = ROTR (F1 (temp[4], md[0], temp[3], temp[6], temp[7], temp[5], temp[2]), 7) + ROTR (md[1], 11) + W[ 6];
	temp[0] = ROTR (F1 (temp[3], temp[7], temp[2], temp[5], temp[6], temp[4], temp[1]), 7) + ROTR (md[0], 11) + W[ 7];

	temp[7] = ROTR (F1 (temp[2], temp[6], temp[1], temp[4], temp[5], temp[3], temp[0]), 7) + ROTR (temp[7], 11) + W[ 8];
	temp[6] = ROTR (F1 (temp[1], temp[5], temp[0], temp[3], temp[4], temp[2], temp[7]), 7) + ROTR (temp[6], 11) + W[ 9];
	temp[5] = ROTR (F1 (temp[0], temp[4], temp[7], temp[2], temp[3], temp[1], temp[6]), 7) + ROTR (temp[5], 11) + W[10];
	temp[4] = ROTR (F1 (temp[7], temp[3], temp[6], temp[1], temp[2], temp[0], temp[5]), 7) + ROTR (temp[4], 11) + W[11];
	temp[3] = ROTR (F1 (temp[6], temp[2], temp[5], temp[0], temp[1], temp[7], temp[4]), 7) + ROTR (temp[3], 11) + W[12];
	temp[2] = ROTR (F1 (temp[5], temp[1], temp[4], temp[7], temp[0], temp[6], temp[3]), 7) + ROTR (temp[2], 11) + W[13];
	temp[1] = ROTR (F1 (temp[4], temp[0], temp[3], temp[6], temp[7], temp[5], temp[2]), 7) + ROTR (temp[1], 11) + W[14];
	temp[0] = ROTR (F1 (temp[3], temp[7], temp[2], temp[5], temp[6], temp[4], temp[1]), 7) + ROTR (temp[0], 11) + W[15];

	temp[7] = ROTR (F1 (temp[2], temp[6], temp[1], temp[4], temp[5], temp[3], temp[0]), 7) + ROTR (temp[7], 11) + W[16];
	temp[6] = ROTR (F1 (temp[1], temp[5], temp[0], temp[3], temp[4], temp[2], temp[7]), 7) + ROTR (temp[6], 11) + W[17];
	temp[5] = ROTR (F1 (temp[0], temp[4], temp[7], temp[2], temp[3], temp[1], temp[6]), 7) + ROTR (temp[5], 11) + W[18];
	temp[4] = ROTR (F1 (temp[7], temp[3], temp[6], temp[1], temp[2], temp[0], temp[5]), 7) + ROTR (temp[4], 11) + W[19];
	temp[3] = ROTR (F1 (temp[6], temp[2], temp[5], temp[0], temp[1], temp[7], temp[4]), 7) + ROTR (temp[3], 11) + W[20];
	temp[2] = ROTR (F1 (temp[5], temp[1], temp[4], temp[7], temp[0], temp[6], temp[3]), 7) + ROTR (temp[2], 11) + W[21];
	temp[1] = ROTR (F1 (temp[4], temp[0], temp[3], temp[6], temp[7], temp[5], temp[2]), 7) + ROTR (temp[1], 11) + W[22];
	temp[0] = ROTR (F1 (temp[3], temp[7], temp[2], temp[5], temp[6], temp[4], temp[1]), 7) + ROTR (temp[0], 11) + W[23];

	temp[7] = ROTR (F1 (temp[2], temp[6], temp[1], temp[4], temp[5], temp[3], temp[0]), 7) + ROTR (temp[7], 11) + W[24];
	temp[6] = ROTR (F1 (temp[1], temp[5], temp[0], temp[3], temp[4], temp[2], temp[7]), 7) + ROTR (temp[6], 11) + W[25];
	temp[5] = ROTR (F1 (temp[0], temp[4], temp[7], temp[2], temp[3], temp[1], temp[6]), 7) + ROTR (temp[5], 11) + W[26];
	temp[4] = ROTR (F1 (temp[7], temp[3], temp[6], temp[1], temp[2], temp[0], temp[5]), 7) + ROTR (temp[4], 11) + W[27];
	temp[3] = ROTR (F1 (temp[6], temp[2], temp[5], temp[0], temp[1], temp[7], temp[4]), 7) + ROTR (temp[3], 11) + W[28];
	temp[2] = ROTR (F1 (temp[5], temp[1], temp[4], temp[7], temp[0], temp[6], temp[3]), 7) + ROTR (temp[2], 11) + W[29];
	temp[1] = ROTR (F1 (temp[4], temp[0], temp[3], temp[6], temp[7], temp[5], temp[2]), 7) + ROTR (temp[1], 11) + W[30];
	temp[0] = ROTR (F1 (temp[3], temp[7], temp[2], temp[5], temp[6], temp[4], temp[1]), 7) + ROTR (temp[0], 11) + W[31];

	/* PASS 2: */

	temp[7] = ROTR (F2 (temp[3], temp[5], temp[2], temp[0], temp[1], temp[6], temp[4]), 7) + ROTR (temp[7], 11) + W[ 5] + 0x452821E6;
	temp[6] = ROTR (F2 (temp[2], temp[4], temp[1], temp[7], temp[0], temp[5], temp[3]), 7) + ROTR (temp[6], 11) + W[14] + 0x38D01377;
	temp[5] = ROTR (F2 (temp[1], temp[3], temp[0], temp[6], temp[7], temp[4], temp[2]), 7) + ROTR (temp[5], 11) + W[26] + 0xBE5466CF;
	temp[4] = ROTR (F2 (temp[0], temp[2], temp[7], temp[5], temp[6], temp[3], temp[1]), 7) + ROTR (temp[4], 11) + W[18] + 0x34E90C6C;
	temp[3] = ROTR (F2 (temp[7], temp[1], temp[6], temp[4], temp[5], temp[2], temp[0]), 7) + ROTR (temp[3], 11) + W[11] + 0xC0AC29B7;
	temp[2] = ROTR (F2 (temp[6], temp[0], temp[5], temp[3], temp[4], temp[1], temp[7]), 7) + ROTR (temp[2], 11) + W[28] + 0xC97C50DD;
	temp[1] = ROTR (F2 (temp[5], temp[7], temp[4], temp[2], temp[3], temp[0], temp[6]), 7) + ROTR (temp[1], 11) + W[ 7] + 0x3F84D5B5;
	temp[0] = ROTR (F2 (temp[4], temp[6], temp[3], temp[1], temp[2], temp[7], temp[5]), 7) + ROTR (temp[0], 11) + W[16] + 0xB5470917;

	temp[7] = ROTR (F2 (temp[3], temp[5], temp[2], temp[0], temp[1], temp[6], temp[4]), 7) + ROTR (temp[7], 11) + W[ 0] + 0x9216D5D9;
	temp[6] = ROTR (F2 (temp[2], temp[4], temp[1], temp[7], temp[0], temp[5], temp[3]), 7) + ROTR (temp[6], 11) + W[23] + 0x8979FB1B;
	temp[5] = ROTR (F2 (temp[1], temp[3], temp[0], temp[6], temp[7], temp[4], temp[2]), 7) + ROTR (temp[5], 11) + W[20] + 0xD1310BA6;
	temp[4] = ROTR (F2 (temp[0], temp[2], temp[7], temp[5], temp[6], temp[3], temp[1]), 7) + ROTR (temp[4], 11) + W[22] + 0x98DFB5AC;
	temp[3] = ROTR (F2 (temp[7], temp[1], temp[6], temp[4], temp[5], temp[2], temp[0]), 7) + ROTR (temp[3], 11) + W[ 1] + 0x2FFD72DB;
	temp[2] = ROTR (F2 (temp[6], temp[0], temp[5], temp[3], temp[4], temp[1], temp[7]), 7) + ROTR (temp[2], 11) + W[10] + 0xD01ADFB7;
	temp[1] = ROTR (F2 (temp[5], temp[7], temp[4], temp[2], temp[3], temp[0], temp[6]), 7) + ROTR (temp[1], 11) + W[ 4] + 0xB8E1AFED;
	temp[0] = ROTR (F2 (temp[4], temp[6], temp[3], temp[1], temp[2], temp[7], temp[5]), 7) + ROTR (temp[0], 11) + W[ 8] + 0x6A267E96;

	temp[7] = ROTR (F2 (temp[3], temp[5], temp[2], temp[0], temp[1], temp[6], temp[4]), 7) + ROTR (temp[7], 11) + W[30] + 0xBA7C9045;
	temp[6] = ROTR (F2 (temp[2], temp[4], temp[1], temp[7], temp[0], temp[5], temp[3]), 7) + ROTR (temp[6], 11) + W[ 3] + 0xF12C7F99;
	temp[5] = ROTR (F2 (temp[1], temp[3], temp[0], temp[6], temp[7], temp[4], temp[2]), 7) + ROTR (temp[5], 11) + W[21] + 0x24A19947;
	temp[4] = ROTR (F2 (temp[0], temp[2], temp[7], temp[5], temp[6], temp[3], temp[1]), 7) + ROTR (temp[4], 11) + W[ 9] + 0xB3916CF7;
	temp[3] = ROTR (F2 (temp[7], temp[1], temp[6], temp[4], temp[5], temp[2], temp[0]), 7) + ROTR (temp[3], 11) + W[17] + 0x0801F2E2;
	temp[2] = ROTR (F2 (temp[6], temp[0], temp[5], temp[3], temp[4], temp[1], temp[7]), 7) + ROTR (temp[2], 11) + W[24] + 0x858EFC16;
	temp[1] = ROTR (F2 (temp[5], temp[7], temp[4], temp[2], temp[3], temp[0], temp[6]), 7) + ROTR (temp[1], 11) + W[29] + 0x636920D8;
	temp[0] = ROTR (F2 (temp[4], temp[6], temp[3], temp[1], temp[2], temp[7], temp[5]), 7) + ROTR (temp[0], 11) + W[ 6] + 0x71574E69;

	temp[7] = ROTR (F2 (temp[3], temp[5], temp[2], temp[0], temp[1], temp[6], temp[4]), 7) + ROTR (temp[7], 11) + W[19] + 0xA458FEA3;
	temp[6] = ROTR (F2 (temp[2], temp[4], temp[1], temp[7], temp[0], temp[5], temp[3]), 7) + ROTR (temp[6], 11) + W[12] + 0xF4933D7E;
	temp[5] = ROTR (F2 (temp[1], temp[3], temp[0], temp[6], temp[7], temp[4], temp[2]), 7) + ROTR (temp[5], 11) + W[15] + 0x0D95748F;
	temp[4] = ROTR (F2 (temp[0], temp[2], temp[7], temp[5], temp[6], temp[3], temp[1]), 7) + ROTR (temp[4], 11) + W[13] + 0x728EB658;
	temp[3] = ROTR (F2 (temp[7], temp[1], temp[6], temp[4], temp[5], temp[2], temp[0]), 7) + ROTR (temp[3], 11) + W[ 2] + 0x718BCD58;
	temp[2] = ROTR (F2 (temp[6], temp[0], temp[5], temp[3], temp[4], temp[1], temp[7]), 7) + ROTR (temp[2], 11) + W[25] + 0x82154AEE;
	temp[1] = ROTR (F2 (temp[5], temp[7], temp[4], temp[2], temp[3], temp[0], temp[6]), 7) + ROTR (temp[1], 11) + W[31] + 0x7B54A41D;
	temp[0] = ROTR (F2 (temp[4], temp[6], temp[3], temp[1], temp[2], temp[7], temp[5]), 7) + ROTR (temp[0], 11) + W[27] + 0xC25A59B5;

	/* PASS 3: */

	temp[7] = ROTR (F3 (temp[1], temp[4], temp[3], temp[6], temp[0], temp[2], temp[5]), 7) + ROTR (temp[7], 11) + W[19] + 0x9C30D539;
	temp[6] = ROTR (F3 (temp[0], temp[3], temp[2], temp[5], temp[7], temp[1], temp[4]), 7) + ROTR (temp[6], 11) + W[ 9] + 0x2AF26013;
	temp[5] = ROTR (F3 (temp[7], temp[2], temp[1], temp[4], temp[6], temp[0], temp[3]), 7) + ROTR (temp[5], 11) + W[ 4] + 0xC5D1B023;
	temp[4] = ROTR (F3 (temp[6], temp[1], temp[0], temp[3], temp[5], temp[7], temp[2]), 7) + ROTR (temp[4], 11) + W[20] + 0x286085F0;
	temp[3] = ROTR (F3 (temp[5], temp[0], temp[7], temp[2], temp[4], temp[6], temp[1]), 7) + ROTR (temp[3], 11) + W[28] + 0xCA417918;
	temp[2] = ROTR (F3 (temp[4], temp[7], temp[6], temp[1], temp[3], temp[5], temp[0]), 7) + ROTR (temp[2], 11) + W[17] + 0xB8DB38EF;
	temp[1] = ROTR (F3 (temp[3], temp[6], temp[5], temp[0], temp[2], temp[4], temp[7]), 7) + ROTR (temp[1], 11) + W[ 8] + 0x8E79DCB0;
	temp[0] = ROTR (F3 (temp[2], temp[5], temp[4], temp[7], temp[1], temp[3], temp[6]), 7) + ROTR (temp[0], 11) + W[22] + 0x603A180E;

	temp[7] = ROTR (F3 (temp[1], temp[4], temp[3], temp[6], temp[0], temp[2], temp[5]), 7) + ROTR (temp[7], 11) + W[29] + 0x6C9E0E8B;
	temp[6] = ROTR (F3 (temp[0], temp[3], temp[2], temp[5], temp[7], temp[1], temp[4]), 7) + ROTR (temp[6], 11) + W[14] + 0xB01E8A3E;
	temp[5] = ROTR (F3 (temp[7], temp[2], temp[1], temp[4], temp[6], temp[0], temp[3]), 7) + ROTR (temp[5], 11) + W[25] + 0xD71577C1;
	temp[4] = ROTR (F3 (temp[6], temp[1], temp[0], temp[3], temp[5], temp[7], temp[2]), 7) + ROTR (temp[4], 11) + W[12] + 0xBD314B27;
	temp[3] = ROTR (F3 (temp[5], temp[0], temp[7], temp[2], temp[4], temp[6], temp[1]), 7) + ROTR (temp[3], 11) + W[24] + 0x78AF2FDA;
	temp[2] = ROTR (F3 (temp[4], temp[7], temp[6], temp[1], temp[3], temp[5], temp[0]), 7) + ROTR (temp[2], 11) + W[30] + 0x55605C60;
	temp[1] = ROTR (F3 (temp[3], temp[6], temp[5], temp[0], temp[2], temp[4], temp[7]), 7) + ROTR (temp[1], 11) + W[16] + 0xE65525F3;
	temp[0] = ROTR (F3 (temp[2], temp[5], temp[4], temp[7], temp[1], temp[3], temp[6]), 7) + ROTR (temp[0], 11) + W[26] + 0xAA55AB94;

	temp[7] = ROTR (F3 (temp[1], temp[4], temp[3], temp[6], temp[0], temp[2], temp[5]), 7) + ROTR (temp[7], 11) + W[31] + 0x57489862;
	temp[6] = ROTR (F3 (temp[0], temp[3], temp[2], temp[5], temp[7], temp[1], temp[4]), 7) + ROTR (temp[6], 11) + W[15] + 0x63E81440;
	temp[5] = ROTR (F3 (temp[7], temp[2], temp[1], temp[4], temp[6], temp[0], temp[3]), 7) + ROTR (temp[5], 11) + W[ 7] + 0x55CA396A;
	temp[4] = ROTR (F3 (temp[6], temp[1], temp[0], temp[3], temp[5], temp[7], temp[2]), 7) + ROTR (temp[4], 11) + W[ 3] + 0x2AAB10B6;
	temp[3] = ROTR (F3 (temp[5], temp[0], temp[7], temp[2], temp[4], temp[6], temp[1]), 7) + ROTR (temp[3], 11) + W[ 1] + 0xB4CC5C34;
	temp[2] = ROTR (F3 (temp[4], temp[7], temp[6], temp[1], temp[3], temp[5], temp[0]), 7) + ROTR (temp[2], 11) + W[ 0] + 0x1141E8CE;
	temp[1] = ROTR (F3 (temp[3], temp[6], temp[5], temp[0], temp[2], temp[4], temp[7]), 7) + ROTR (temp[1], 11) + W[18] + 0xA15486AF;
	temp[0] = ROTR (F3 (temp[2], temp[5], temp[4], temp[7], temp[1], temp[3], temp[6]), 7) + ROTR (temp[0], 11) + W[27] + 0x7C72E993;

	temp[7] = ROTR (F3 (temp[1], temp[4], temp[3], temp[6], temp[0], temp[2], temp[5]), 7) + ROTR (temp[7], 11) + W[13] + 0xB3EE1411;
	temp[6] = ROTR (F3 (temp[0], temp[3], temp[2], temp[5], temp[7], temp[1], temp[4]), 7) + ROTR (temp[6], 11) + W[ 6] + 0x636FBC2A;
	temp[5] = ROTR (F3 (temp[7], temp[2], temp[1], temp[4], temp[6], temp[0], temp[3]), 7) + ROTR (temp[5], 11) + W[21] + 0x2BA9C55D;
	temp[4] = ROTR (F3 (temp[6], temp[1], temp[0], temp[3], temp[5], temp[7], temp[2]), 7) + ROTR (temp[4], 11) + W[10] + 0x741831F6;
	temp[3] = ROTR (F3 (temp[5], temp[0], temp[7], temp[2], temp[4], temp[6], temp[1]), 7) + ROTR (temp[3], 11) + W[23] + 0xCE5C3E16;
	temp[2] = ROTR (F3 (temp[4], temp[7], temp[6], temp[1], temp[3], temp[5], temp[0]), 7) + ROTR (temp[2], 11) + W[11] + 0x9B87931E;
	temp[1] = ROTR (F3 (temp[3], temp[6], temp[5], temp[0], temp[2], temp[4], temp[7]), 7) + ROTR (temp[1], 11) + W[ 5] + 0xAFD6BA33;
	temp[0] = ROTR (F3 (temp[2], temp[5], temp[4], temp[7], temp[1], temp[3], temp[6]), 7) + ROTR (temp[0], 11) + W[ 2] + 0x6C24CF5C;

	/* PASS 4: */

	temp[7] = ROTR (F4 (temp[6], temp[4], temp[0], temp[5], temp[2], temp[1], temp[3]), 7) + ROTR (temp[7], 11) + W[24] + 0x7A325381;
	temp[6] = ROTR (F4 (temp[5], temp[3], temp[7], temp[4], temp[1], temp[0], temp[2]), 7) + ROTR (temp[6], 11) + W[ 4] + 0x28958677;
	temp[5] = ROTR (F4 (temp[4], temp[2], temp[6], temp[3], temp[0], temp[7], temp[1]), 7) + ROTR (temp[5], 11) + W[ 0] + 0x3B8F4898;
	temp[4] = ROTR (F4 (temp[3], temp[1], temp[5], temp[2], temp[7], temp[6], temp[0]), 7) + ROTR (temp[4], 11) + W[14] + 0x6B4BB9AF;
	temp[3] = ROTR (F4 (temp[2], temp[0], temp[4], temp[1], temp[6], temp[5], temp[7]), 7) + ROTR (temp[3], 11) + W[ 2] + 0xC4BFE81B;
	temp[2] = ROTR (F4 (temp[1], temp[7], temp[3], temp[0], temp[5], temp[4], temp[6]), 7) + ROTR (temp[2], 11) + W[ 7] + 0x66282193;
	temp[1] = ROTR (F4 (temp[0], temp[6], temp[2], temp[7], temp[4], temp[3], temp[5]), 7) + ROTR (temp[1], 11) + W[28] + 0x61D809CC;
	temp[0] = ROTR (F4 (temp[7], temp[5], temp[1], temp[6], temp[3], temp[2], temp[4]), 7) + ROTR (temp[0], 11) + W[23] + 0xFB21A991;

	temp[7] = ROTR (F4 (temp[6], temp[4], temp[0], temp[5], temp[2], temp[1], temp[3]), 7) + ROTR (temp[7], 11) + W[26] + 0x487CAC60;
	temp[6] = ROTR (F4 (temp[5], temp[3], temp[7], temp[4], temp[1], temp[0], temp[2]), 7) + ROTR (temp[6], 11) + W[ 6] + 0x5DEC8032;
	temp[5] = ROTR (F4 (temp[4], temp[2], temp[6], temp[3], temp[0], temp[7], temp[1]), 7) + ROTR (temp[5], 11) + W[30] + 0xEF845D5D;
	temp[4] = ROTR (F4 (temp[3], temp[1], temp[5], temp[2], temp[7], temp[6], temp[0]), 7) + ROTR (temp[4], 11) + W[20] + 0xE98575B1;
	temp[3] = ROTR (F4 (temp[2], temp[0], temp[4], temp[1], temp[6], temp[5], temp[7]), 7) + ROTR (temp[3], 11) + W[18] + 0xDC262302;
	temp[2] = ROTR (F4 (temp[1], temp[7], temp[3], temp[0], temp[5], temp[4], temp[6]), 7) + ROTR (temp[2], 11) + W[25] + 0xEB651B88;
	temp[1] = ROTR (F4 (temp[0], temp[6], temp[2], temp[7], temp[4], temp[3], temp[5]), 7) + ROTR (temp[1], 11) + W[19] + 0x23893E81;
	temp[0] = ROTR (F4 (temp[7], temp[5], temp[1], temp[6], temp[3], temp[2], temp[4]), 7) + ROTR (temp[0], 11) + W[ 3] + 0xD396ACC5;

	temp[7] = ROTR (F4 (temp[6], temp[4], temp[0], temp[5], temp[2], temp[1], temp[3]), 7) + ROTR (temp[7], 11) + W[22] + 0x0F6D6FF3;
	temp[6] = ROTR (F4 (temp[5], temp[3], temp[7], temp[4], temp[1], temp[0], temp[2]), 7) + ROTR (temp[6], 11) + W[11] + 0x83F44239;
	temp[5] = ROTR (F4 (temp[4], temp[2], temp[6], temp[3], temp[0], temp[7], temp[1]), 7) + ROTR (temp[5], 11) + W[31] + 0x2E0B4482;
	temp[4] = ROTR (F4 (temp[3], temp[1], temp[5], temp[2], temp[7], temp[6], temp[0]), 7) + ROTR (temp[4], 11) + W[21] + 0xA4842004;
	temp[3] = ROTR (F4 (temp[2], temp[0], temp[4], temp[1], temp[6], temp[5], temp[7]), 7) + ROTR (temp[3], 11) + W[ 8] + 0x69C8F04A;
	temp[2] = ROTR (F4 (temp[1], temp[7], temp[3], temp[0], temp[5], temp[4], temp[6]), 7) + ROTR (temp[2], 11) + W[27] + 0x9E1F9B5E;
	temp[1] = ROTR (F4 (temp[0], temp[6], temp[2], temp[7], temp[4], temp[3], temp[5]), 7) + ROTR (temp[1], 11) + W[12] + 0x21C66842;
	temp[0] = ROTR (F4 (temp[7], temp[5], temp[1], temp[6], temp[3], temp[2], temp[4]), 7) + ROTR (temp[0], 11) + W[ 9] + 0xF6E96C9A;

	md[7] += temp[7] = ROTR (F4 (temp[6], temp[4], temp[0], temp[5], temp[2], temp[1], temp[3]), 7) + ROTR (temp[7], 11) + W[ 1] + 0x670C9C61;
	md[6] += temp[6] = ROTR (F4 (temp[5], temp[3], temp[7], temp[4], temp[1], temp[0], temp[2]), 7) + ROTR (temp[6], 11) + W[29] + 0xABD388F0;
	md[5] += temp[5] = ROTR (F4 (temp[4], temp[2], temp[6], temp[3], temp[0], temp[7], temp[1]), 7) + ROTR (temp[5], 11) + W[ 5] + 0x6A51A0D2;
	md[4] += temp[4] = ROTR (F4 (temp[3], temp[1], temp[5], temp[2], temp[7], temp[6], temp[0]), 7) + ROTR (temp[4], 11) + W[15] + 0xD8542F68;
	md[3] += temp[3] = ROTR (F4 (temp[2], temp[0], temp[4], temp[1], temp[6], temp[5], temp[7]), 7) + ROTR (temp[3], 11) + W[17] + 0x960FA728;
	md[2] += temp[2] = ROTR (F4 (temp[1], temp[7], temp[3], temp[0], temp[5], temp[4], temp[6]), 7) + ROTR (temp[2], 11) + W[10] + 0xAB5133A3;
	md[1] += temp[1] = ROTR (F4 (temp[0], temp[6], temp[2], temp[7], temp[4], temp[3], temp[5]), 7) + ROTR (temp[1], 11) + W[16] + 0x6EEF0B6C;
	md[0] += temp[0] = ROTR (F4 (temp[7], temp[5], temp[1], temp[6], temp[3], temp[2], temp[4]), 7) + ROTR (temp[0], 11) + W[13] + 0x137A3BE4;

        for(i=0; i<32; ++i)W[i] = 0;
    }

    void transform5 (byte[] D, int offset)
    {
        int[] W = new int[32];
        int i;
        for(i=0; i<32; ++i)
        {
            W[i] =
                ((D[offset+3+4*i]&0xFF)<<24) |
                ((D[offset+2+4*i]&0xFF)<<16) |
                ((D[offset+1+4*i]&0xFF)<< 8) |
                ((D[offset+0+4*i]&0xFF)    );
        }

	/* PASS 1: */

	temp[7] = ROTR (F1 (md[3], md[4], md[1], md[0], md[5], md[2], md[6]), 7) + ROTR (md[7], 11) + W[ 0];
	temp[6] = ROTR (F1 (md[2], md[3], md[0], temp[7], md[4], md[1], md[5]), 7) + ROTR (md[6], 11) + W[ 1];
	temp[5] = ROTR (F1 (md[1], md[2], temp[7], temp[6], md[3], md[0], md[4]), 7) + ROTR (md[5], 11) + W[ 2];
	temp[4] = ROTR (F1 (md[0], md[1], temp[6], temp[5], md[2], temp[7], md[3]), 7) + ROTR (md[4], 11) + W[ 3];
	temp[3] = ROTR (F1 (temp[7], md[0], temp[5], temp[4], md[1], temp[6], md[2]), 7) + ROTR (md[3], 11) + W[ 4];
	temp[2] = ROTR (F1 (temp[6], temp[7], temp[4], temp[3], md[0], temp[5], md[1]), 7) + ROTR (md[2], 11) + W[ 5];
	temp[1] = ROTR (F1 (temp[5], temp[6], temp[3], temp[2], temp[7], temp[4], md[0]), 7) + ROTR (md[1], 11) + W[ 6];
	temp[0] = ROTR (F1 (temp[4], temp[5], temp[2], temp[1], temp[6], temp[3], temp[7]), 7) + ROTR (md[0], 11) + W[ 7];

	temp[7] = ROTR (F1 (temp[3], temp[4], temp[1], temp[0], temp[5], temp[2], temp[6]), 7) + ROTR (temp[7], 11) + W[ 8];
	temp[6] = ROTR (F1 (temp[2], temp[3], temp[0], temp[7], temp[4], temp[1], temp[5]), 7) + ROTR (temp[6], 11) + W[ 9];
	temp[5] = ROTR (F1 (temp[1], temp[2], temp[7], temp[6], temp[3], temp[0], temp[4]), 7) + ROTR (temp[5], 11) + W[10];
	temp[4] = ROTR (F1 (temp[0], temp[1], temp[6], temp[5], temp[2], temp[7], temp[3]), 7) + ROTR (temp[4], 11) + W[11];
	temp[3] = ROTR (F1 (temp[7], temp[0], temp[5], temp[4], temp[1], temp[6], temp[2]), 7) + ROTR (temp[3], 11) + W[12];
	temp[2] = ROTR (F1 (temp[6], temp[7], temp[4], temp[3], temp[0], temp[5], temp[1]), 7) + ROTR (temp[2], 11) + W[13];
	temp[1] = ROTR (F1 (temp[5], temp[6], temp[3], temp[2], temp[7], temp[4], temp[0]), 7) + ROTR (temp[1], 11) + W[14];
	temp[0] = ROTR (F1 (temp[4], temp[5], temp[2], temp[1], temp[6], temp[3], temp[7]), 7) + ROTR (temp[0], 11) + W[15];

	temp[7] = ROTR (F1 (temp[3], temp[4], temp[1], temp[0], temp[5], temp[2], temp[6]), 7) + ROTR (temp[7], 11) + W[16];
	temp[6] = ROTR (F1 (temp[2], temp[3], temp[0], temp[7], temp[4], temp[1], temp[5]), 7) + ROTR (temp[6], 11) + W[17];
	temp[5] = ROTR (F1 (temp[1], temp[2], temp[7], temp[6], temp[3], temp[0], temp[4]), 7) + ROTR (temp[5], 11) + W[18];
	temp[4] = ROTR (F1 (temp[0], temp[1], temp[6], temp[5], temp[2], temp[7], temp[3]), 7) + ROTR (temp[4], 11) + W[19];
	temp[3] = ROTR (F1 (temp[7], temp[0], temp[5], temp[4], temp[1], temp[6], temp[2]), 7) + ROTR (temp[3], 11) + W[20];
	temp[2] = ROTR (F1 (temp[6], temp[7], temp[4], temp[3], temp[0], temp[5], temp[1]), 7) + ROTR (temp[2], 11) + W[21];
	temp[1] = ROTR (F1 (temp[5], temp[6], temp[3], temp[2], temp[7], temp[4], temp[0]), 7) + ROTR (temp[1], 11) + W[22];
	temp[0] = ROTR (F1 (temp[4], temp[5], temp[2], temp[1], temp[6], temp[3], temp[7]), 7) + ROTR (temp[0], 11) + W[23];

	temp[7] = ROTR (F1 (temp[3], temp[4], temp[1], temp[0], temp[5], temp[2], temp[6]), 7) + ROTR (temp[7], 11) + W[24];
	temp[6] = ROTR (F1 (temp[2], temp[3], temp[0], temp[7], temp[4], temp[1], temp[5]), 7) + ROTR (temp[6], 11) + W[25];
	temp[5] = ROTR (F1 (temp[1], temp[2], temp[7], temp[6], temp[3], temp[0], temp[4]), 7) + ROTR (temp[5], 11) + W[26];
	temp[4] = ROTR (F1 (temp[0], temp[1], temp[6], temp[5], temp[2], temp[7], temp[3]), 7) + ROTR (temp[4], 11) + W[27];
	temp[3] = ROTR (F1 (temp[7], temp[0], temp[5], temp[4], temp[1], temp[6], temp[2]), 7) + ROTR (temp[3], 11) + W[28];
	temp[2] = ROTR (F1 (temp[6], temp[7], temp[4], temp[3], temp[0], temp[5], temp[1]), 7) + ROTR (temp[2], 11) + W[29];
	temp[1] = ROTR (F1 (temp[5], temp[6], temp[3], temp[2], temp[7], temp[4], temp[0]), 7) + ROTR (temp[1], 11) + W[30];
	temp[0] = ROTR (F1 (temp[4], temp[5], temp[2], temp[1], temp[6], temp[3], temp[7]), 7) + ROTR (temp[0], 11) + W[31];

	/* PASS 2: */

	temp[7] = ROTR (F2 (temp[6], temp[2], temp[1], temp[0], temp[3], temp[4], temp[5]), 7) + ROTR (temp[7], 11) + W[ 5] + 0x452821E6;
	temp[6] = ROTR (F2 (temp[5], temp[1], temp[0], temp[7], temp[2], temp[3], temp[4]), 7) + ROTR (temp[6], 11) + W[14] + 0x38D01377;
	temp[5] = ROTR (F2 (temp[4], temp[0], temp[7], temp[6], temp[1], temp[2], temp[3]), 7) + ROTR (temp[5], 11) + W[26] + 0xBE5466CF;
	temp[4] = ROTR (F2 (temp[3], temp[7], temp[6], temp[5], temp[0], temp[1], temp[2]), 7) + ROTR (temp[4], 11) + W[18] + 0x34E90C6C;
	temp[3] = ROTR (F2 (temp[2], temp[6], temp[5], temp[4], temp[7], temp[0], temp[1]), 7) + ROTR (temp[3], 11) + W[11] + 0xC0AC29B7;
	temp[2] = ROTR (F2 (temp[1], temp[5], temp[4], temp[3], temp[6], temp[7], temp[0]), 7) + ROTR (temp[2], 11) + W[28] + 0xC97C50DD;
	temp[1] = ROTR (F2 (temp[0], temp[4], temp[3], temp[2], temp[5], temp[6], temp[7]), 7) + ROTR (temp[1], 11) + W[ 7] + 0x3F84D5B5;
	temp[0] = ROTR (F2 (temp[7], temp[3], temp[2], temp[1], temp[4], temp[5], temp[6]), 7) + ROTR (temp[0], 11) + W[16] + 0xB5470917;

	temp[7] = ROTR (F2 (temp[6], temp[2], temp[1], temp[0], temp[3], temp[4], temp[5]), 7) + ROTR (temp[7], 11) + W[ 0] + 0x9216D5D9;
	temp[6] = ROTR (F2 (temp[5], temp[1], temp[0], temp[7], temp[2], temp[3], temp[4]), 7) + ROTR (temp[6], 11) + W[23] + 0x8979FB1B;
	temp[5] = ROTR (F2 (temp[4], temp[0], temp[7], temp[6], temp[1], temp[2], temp[3]), 7) + ROTR (temp[5], 11) + W[20] + 0xD1310BA6;
	temp[4] = ROTR (F2 (temp[3], temp[7], temp[6], temp[5], temp[0], temp[1], temp[2]), 7) + ROTR (temp[4], 11) + W[22] + 0x98DFB5AC;
	temp[3] = ROTR (F2 (temp[2], temp[6], temp[5], temp[4], temp[7], temp[0], temp[1]), 7) + ROTR (temp[3], 11) + W[ 1] + 0x2FFD72DB;
	temp[2] = ROTR (F2 (temp[1], temp[5], temp[4], temp[3], temp[6], temp[7], temp[0]), 7) + ROTR (temp[2], 11) + W[10] + 0xD01ADFB7;
	temp[1] = ROTR (F2 (temp[0], temp[4], temp[3], temp[2], temp[5], temp[6], temp[7]), 7) + ROTR (temp[1], 11) + W[ 4] + 0xB8E1AFED;
	temp[0] = ROTR (F2 (temp[7], temp[3], temp[2], temp[1], temp[4], temp[5], temp[6]), 7) + ROTR (temp[0], 11) + W[ 8] + 0x6A267E96;

	temp[7] = ROTR (F2 (temp[6], temp[2], temp[1], temp[0], temp[3], temp[4], temp[5]), 7) + ROTR (temp[7], 11) + W[30] + 0xBA7C9045;
	temp[6] = ROTR (F2 (temp[5], temp[1], temp[0], temp[7], temp[2], temp[3], temp[4]), 7) + ROTR (temp[6], 11) + W[ 3] + 0xF12C7F99;
	temp[5] = ROTR (F2 (temp[4], temp[0], temp[7], temp[6], temp[1], temp[2], temp[3]), 7) + ROTR (temp[5], 11) + W[21] + 0x24A19947;
	temp[4] = ROTR (F2 (temp[3], temp[7], temp[6], temp[5], temp[0], temp[1], temp[2]), 7) + ROTR (temp[4], 11) + W[ 9] + 0xB3916CF7;
	temp[3] = ROTR (F2 (temp[2], temp[6], temp[5], temp[4], temp[7], temp[0], temp[1]), 7) + ROTR (temp[3], 11) + W[17] + 0x0801F2E2;
	temp[2] = ROTR (F2 (temp[1], temp[5], temp[4], temp[3], temp[6], temp[7], temp[0]), 7) + ROTR (temp[2], 11) + W[24] + 0x858EFC16;
	temp[1] = ROTR (F2 (temp[0], temp[4], temp[3], temp[2], temp[5], temp[6], temp[7]), 7) + ROTR (temp[1], 11) + W[29] + 0x636920D8;
	temp[0] = ROTR (F2 (temp[7], temp[3], temp[2], temp[1], temp[4], temp[5], temp[6]), 7) + ROTR (temp[0], 11) + W[ 6] + 0x71574E69;

	temp[7] = ROTR (F2 (temp[6], temp[2], temp[1], temp[0], temp[3], temp[4], temp[5]), 7) + ROTR (temp[7], 11) + W[19] + 0xA458FEA3;
	temp[6] = ROTR (F2 (temp[5], temp[1], temp[0], temp[7], temp[2], temp[3], temp[4]), 7) + ROTR (temp[6], 11) + W[12] + 0xF4933D7E;
	temp[5] = ROTR (F2 (temp[4], temp[0], temp[7], temp[6], temp[1], temp[2], temp[3]), 7) + ROTR (temp[5], 11) + W[15] + 0x0D95748F;
	temp[4] = ROTR (F2 (temp[3], temp[7], temp[6], temp[5], temp[0], temp[1], temp[2]), 7) + ROTR (temp[4], 11) + W[13] + 0x728EB658;
	temp[3] = ROTR (F2 (temp[2], temp[6], temp[5], temp[4], temp[7], temp[0], temp[1]), 7) + ROTR (temp[3], 11) + W[ 2] + 0x718BCD58;
	temp[2] = ROTR (F2 (temp[1], temp[5], temp[4], temp[3], temp[6], temp[7], temp[0]), 7) + ROTR (temp[2], 11) + W[25] + 0x82154AEE;
	temp[1] = ROTR (F2 (temp[0], temp[4], temp[3], temp[2], temp[5], temp[6], temp[7]), 7) + ROTR (temp[1], 11) + W[31] + 0x7B54A41D;
	temp[0] = ROTR (F2 (temp[7], temp[3], temp[2], temp[1], temp[4], temp[5], temp[6]), 7) + ROTR (temp[0], 11) + W[27] + 0xC25A59B5;

	/* PASS 3: */

	temp[7] = ROTR (F3 (temp[2], temp[6], temp[0], temp[4], temp[3], temp[1], temp[5]), 7) + ROTR (temp[7], 11) + W[19] + 0x9C30D539;
	temp[6] = ROTR (F3 (temp[1], temp[5], temp[7], temp[3], temp[2], temp[0], temp[4]), 7) + ROTR (temp[6], 11) + W[ 9] + 0x2AF26013;
	temp[5] = ROTR (F3 (temp[0], temp[4], temp[6], temp[2], temp[1], temp[7], temp[3]), 7) + ROTR (temp[5], 11) + W[ 4] + 0xC5D1B023;
	temp[4] = ROTR (F3 (temp[7], temp[3], temp[5], temp[1], temp[0], temp[6], temp[2]), 7) + ROTR (temp[4], 11) + W[20] + 0x286085F0;
	temp[3] = ROTR (F3 (temp[6], temp[2], temp[4], temp[0], temp[7], temp[5], temp[1]), 7) + ROTR (temp[3], 11) + W[28] + 0xCA417918;
	temp[2] = ROTR (F3 (temp[5], temp[1], temp[3], temp[7], temp[6], temp[4], temp[0]), 7) + ROTR (temp[2], 11) + W[17] + 0xB8DB38EF;
	temp[1] = ROTR (F3 (temp[4], temp[0], temp[2], temp[6], temp[5], temp[3], temp[7]), 7) + ROTR (temp[1], 11) + W[ 8] + 0x8E79DCB0;
	temp[0] = ROTR (F3 (temp[3], temp[7], temp[1], temp[5], temp[4], temp[2], temp[6]), 7) + ROTR (temp[0], 11) + W[22] + 0x603A180E;

	temp[7] = ROTR (F3 (temp[2], temp[6], temp[0], temp[4], temp[3], temp[1], temp[5]), 7) + ROTR (temp[7], 11) + W[29] + 0x6C9E0E8B;
	temp[6] = ROTR (F3 (temp[1], temp[5], temp[7], temp[3], temp[2], temp[0], temp[4]), 7) + ROTR (temp[6], 11) + W[14] + 0xB01E8A3E;
	temp[5] = ROTR (F3 (temp[0], temp[4], temp[6], temp[2], temp[1], temp[7], temp[3]), 7) + ROTR (temp[5], 11) + W[25] + 0xD71577C1;
	temp[4] = ROTR (F3 (temp[7], temp[3], temp[5], temp[1], temp[0], temp[6], temp[2]), 7) + ROTR (temp[4], 11) + W[12] + 0xBD314B27;
	temp[3] = ROTR (F3 (temp[6], temp[2], temp[4], temp[0], temp[7], temp[5], temp[1]), 7) + ROTR (temp[3], 11) + W[24] + 0x78AF2FDA;
	temp[2] = ROTR (F3 (temp[5], temp[1], temp[3], temp[7], temp[6], temp[4], temp[0]), 7) + ROTR (temp[2], 11) + W[30] + 0x55605C60;
	temp[1] = ROTR (F3 (temp[4], temp[0], temp[2], temp[6], temp[5], temp[3], temp[7]), 7) + ROTR (temp[1], 11) + W[16] + 0xE65525F3;
	temp[0] = ROTR (F3 (temp[3], temp[7], temp[1], temp[5], temp[4], temp[2], temp[6]), 7) + ROTR (temp[0], 11) + W[26] + 0xAA55AB94;

	temp[7] = ROTR (F3 (temp[2], temp[6], temp[0], temp[4], temp[3], temp[1], temp[5]), 7) + ROTR (temp[7], 11) + W[31] + 0x57489862;
	temp[6] = ROTR (F3 (temp[1], temp[5], temp[7], temp[3], temp[2], temp[0], temp[4]), 7) + ROTR (temp[6], 11) + W[15] + 0x63E81440;
	temp[5] = ROTR (F3 (temp[0], temp[4], temp[6], temp[2], temp[1], temp[7], temp[3]), 7) + ROTR (temp[5], 11) + W[ 7] + 0x55CA396A;
	temp[4] = ROTR (F3 (temp[7], temp[3], temp[5], temp[1], temp[0], temp[6], temp[2]), 7) + ROTR (temp[4], 11) + W[ 3] + 0x2AAB10B6;
	temp[3] = ROTR (F3 (temp[6], temp[2], temp[4], temp[0], temp[7], temp[5], temp[1]), 7) + ROTR (temp[3], 11) + W[ 1] + 0xB4CC5C34;
	temp[2] = ROTR (F3 (temp[5], temp[1], temp[3], temp[7], temp[6], temp[4], temp[0]), 7) + ROTR (temp[2], 11) + W[ 0] + 0x1141E8CE;
	temp[1] = ROTR (F3 (temp[4], temp[0], temp[2], temp[6], temp[5], temp[3], temp[7]), 7) + ROTR (temp[1], 11) + W[18] + 0xA15486AF;
	temp[0] = ROTR (F3 (temp[3], temp[7], temp[1], temp[5], temp[4], temp[2], temp[6]), 7) + ROTR (temp[0], 11) + W[27] + 0x7C72E993;

	temp[7] = ROTR (F3 (temp[2], temp[6], temp[0], temp[4], temp[3], temp[1], temp[5]), 7) + ROTR (temp[7], 11) + W[13] + 0xB3EE1411;
	temp[6] = ROTR (F3 (temp[1], temp[5], temp[7], temp[3], temp[2], temp[0], temp[4]), 7) + ROTR (temp[6], 11) + W[ 6] + 0x636FBC2A;
	temp[5] = ROTR (F3 (temp[0], temp[4], temp[6], temp[2], temp[1], temp[7], temp[3]), 7) + ROTR (temp[5], 11) + W[21] + 0x2BA9C55D;
	temp[4] = ROTR (F3 (temp[7], temp[3], temp[5], temp[1], temp[0], temp[6], temp[2]), 7) + ROTR (temp[4], 11) + W[10] + 0x741831F6;
	temp[3] = ROTR (F3 (temp[6], temp[2], temp[4], temp[0], temp[7], temp[5], temp[1]), 7) + ROTR (temp[3], 11) + W[23] + 0xCE5C3E16;
	temp[2] = ROTR (F3 (temp[5], temp[1], temp[3], temp[7], temp[6], temp[4], temp[0]), 7) + ROTR (temp[2], 11) + W[11] + 0x9B87931E;
	temp[1] = ROTR (F3 (temp[4], temp[0], temp[2], temp[6], temp[5], temp[3], temp[7]), 7) + ROTR (temp[1], 11) + W[ 5] + 0xAFD6BA33;
	temp[0] = ROTR (F3 (temp[3], temp[7], temp[1], temp[5], temp[4], temp[2], temp[6]), 7) + ROTR (temp[0], 11) + W[ 2] + 0x6C24CF5C;

	/* PASS 4: */

	temp[7] = ROTR (F4 (temp[1], temp[5], temp[3], temp[2], temp[0], temp[4], temp[6]), 7) + ROTR (temp[7], 11) + W[24] + 0x7A325381;
	temp[6] = ROTR (F4 (temp[0], temp[4], temp[2], temp[1], temp[7], temp[3], temp[5]), 7) + ROTR (temp[6], 11) + W[ 4] + 0x28958677;
	temp[5] = ROTR (F4 (temp[7], temp[3], temp[1], temp[0], temp[6], temp[2], temp[4]), 7) + ROTR (temp[5], 11) + W[ 0] + 0x3B8F4898;
	temp[4] = ROTR (F4 (temp[6], temp[2], temp[0], temp[7], temp[5], temp[1], temp[3]), 7) + ROTR (temp[4], 11) + W[14] + 0x6B4BB9AF;
	temp[3] = ROTR (F4 (temp[5], temp[1], temp[7], temp[6], temp[4], temp[0], temp[2]), 7) + ROTR (temp[3], 11) + W[ 2] + 0xC4BFE81B;
	temp[2] = ROTR (F4 (temp[4], temp[0], temp[6], temp[5], temp[3], temp[7], temp[1]), 7) + ROTR (temp[2], 11) + W[ 7] + 0x66282193;
	temp[1] = ROTR (F4 (temp[3], temp[7], temp[5], temp[4], temp[2], temp[6], temp[0]), 7) + ROTR (temp[1], 11) + W[28] + 0x61D809CC;
	temp[0] = ROTR (F4 (temp[2], temp[6], temp[4], temp[3], temp[1], temp[5], temp[7]), 7) + ROTR (temp[0], 11) + W[23] + 0xFB21A991;

	temp[7] = ROTR (F4 (temp[1], temp[5], temp[3], temp[2], temp[0], temp[4], temp[6]), 7) + ROTR (temp[7], 11) + W[26] + 0x487CAC60;
	temp[6] = ROTR (F4 (temp[0], temp[4], temp[2], temp[1], temp[7], temp[3], temp[5]), 7) + ROTR (temp[6], 11) + W[ 6] + 0x5DEC8032;
	temp[5] = ROTR (F4 (temp[7], temp[3], temp[1], temp[0], temp[6], temp[2], temp[4]), 7) + ROTR (temp[5], 11) + W[30] + 0xEF845D5D;
	temp[4] = ROTR (F4 (temp[6], temp[2], temp[0], temp[7], temp[5], temp[1], temp[3]), 7) + ROTR (temp[4], 11) + W[20] + 0xE98575B1;
	temp[3] = ROTR (F4 (temp[5], temp[1], temp[7], temp[6], temp[4], temp[0], temp[2]), 7) + ROTR (temp[3], 11) + W[18] + 0xDC262302;
	temp[2] = ROTR (F4 (temp[4], temp[0], temp[6], temp[5], temp[3], temp[7], temp[1]), 7) + ROTR (temp[2], 11) + W[25] + 0xEB651B88;
	temp[1] = ROTR (F4 (temp[3], temp[7], temp[5], temp[4], temp[2], temp[6], temp[0]), 7) + ROTR (temp[1], 11) + W[19] + 0x23893E81;
	temp[0] = ROTR (F4 (temp[2], temp[6], temp[4], temp[3], temp[1], temp[5], temp[7]), 7) + ROTR (temp[0], 11) + W[ 3] + 0xD396ACC5;

	temp[7] = ROTR (F4 (temp[1], temp[5], temp[3], temp[2], temp[0], temp[4], temp[6]), 7) + ROTR (temp[7], 11) + W[22] + 0x0F6D6FF3;
	temp[6] = ROTR (F4 (temp[0], temp[4], temp[2], temp[1], temp[7], temp[3], temp[5]), 7) + ROTR (temp[6], 11) + W[11] + 0x83F44239;
	temp[5] = ROTR (F4 (temp[7], temp[3], temp[1], temp[0], temp[6], temp[2], temp[4]), 7) + ROTR (temp[5], 11) + W[31] + 0x2E0B4482;
	temp[4] = ROTR (F4 (temp[6], temp[2], temp[0], temp[7], temp[5], temp[1], temp[3]), 7) + ROTR (temp[4], 11) + W[21] + 0xA4842004;
	temp[3] = ROTR (F4 (temp[5], temp[1], temp[7], temp[6], temp[4], temp[0], temp[2]), 7) + ROTR (temp[3], 11) + W[ 8] + 0x69C8F04A;
	temp[2] = ROTR (F4 (temp[4], temp[0], temp[6], temp[5], temp[3], temp[7], temp[1]), 7) + ROTR (temp[2], 11) + W[27] + 0x9E1F9B5E;
	temp[1] = ROTR (F4 (temp[3], temp[7], temp[5], temp[4], temp[2], temp[6], temp[0]), 7) + ROTR (temp[1], 11) + W[12] + 0x21C66842;
	temp[0] = ROTR (F4 (temp[2], temp[6], temp[4], temp[3], temp[1], temp[5], temp[7]), 7) + ROTR (temp[0], 11) + W[ 9] + 0xF6E96C9A;

	temp[7] = ROTR (F4 (temp[1], temp[5], temp[3], temp[2], temp[0], temp[4], temp[6]), 7) + ROTR (temp[7], 11) + W[ 1] + 0x670C9C61;
	temp[6] = ROTR (F4 (temp[0], temp[4], temp[2], temp[1], temp[7], temp[3], temp[5]), 7) + ROTR (temp[6], 11) + W[29] + 0xABD388F0;
	temp[5] = ROTR (F4 (temp[7], temp[3], temp[1], temp[0], temp[6], temp[2], temp[4]), 7) + ROTR (temp[5], 11) + W[ 5] + 0x6A51A0D2;
	temp[4] = ROTR (F4 (temp[6], temp[2], temp[0], temp[7], temp[5], temp[1], temp[3]), 7) + ROTR (temp[4], 11) + W[15] + 0xD8542F68;
	temp[3] = ROTR (F4 (temp[5], temp[1], temp[7], temp[6], temp[4], temp[0], temp[2]), 7) + ROTR (temp[3], 11) + W[17] + 0x960FA728;
	temp[2] = ROTR (F4 (temp[4], temp[0], temp[6], temp[5], temp[3], temp[7], temp[1]), 7) + ROTR (temp[2], 11) + W[10] + 0xAB5133A3;
	temp[1] = ROTR (F4 (temp[3], temp[7], temp[5], temp[4], temp[2], temp[6], temp[0]), 7) + ROTR (temp[1], 11) + W[16] + 0x6EEF0B6C;
	temp[0] = ROTR (F4 (temp[2], temp[6], temp[4], temp[3], temp[1], temp[5], temp[7]), 7) + ROTR (temp[0], 11) + W[13] + 0x137A3BE4;

	/* PASS 5: */

	temp[7] = ROTR (F5 (temp[2], temp[5], temp[0], temp[6], temp[4], temp[3], temp[1]), 7) + ROTR (temp[7], 11) + W[27] + 0xBA3BF050;
	temp[6] = ROTR (F5 (temp[1], temp[4], temp[7], temp[5], temp[3], temp[2], temp[0]), 7) + ROTR (temp[6], 11) + W[ 3] + 0x7EFB2A98;
	temp[5] = ROTR (F5 (temp[0], temp[3], temp[6], temp[4], temp[2], temp[1], temp[7]), 7) + ROTR (temp[5], 11) + W[21] + 0xA1F1651D;
	temp[4] = ROTR (F5 (temp[7], temp[2], temp[5], temp[3], temp[1], temp[0], temp[6]), 7) + ROTR (temp[4], 11) + W[26] + 0x39AF0176;
	temp[3] = ROTR (F5 (temp[6], temp[1], temp[4], temp[2], temp[0], temp[7], temp[5]), 7) + ROTR (temp[3], 11) + W[17] + 0x66CA593E;
	temp[2] = ROTR (F5 (temp[5], temp[0], temp[3], temp[1], temp[7], temp[6], temp[4]), 7) + ROTR (temp[2], 11) + W[11] + 0x82430E88;
	temp[1] = ROTR (F5 (temp[4], temp[7], temp[2], temp[0], temp[6], temp[5], temp[3]), 7) + ROTR (temp[1], 11) + W[20] + 0x8CEE8619;
	temp[0] = ROTR (F5 (temp[3], temp[6], temp[1], temp[7], temp[5], temp[4], temp[2]), 7) + ROTR (temp[0], 11) + W[29] + 0x456F9FB4;

	temp[7] = ROTR (F5 (temp[2], temp[5], temp[0], temp[6], temp[4], temp[3], temp[1]), 7) + ROTR (temp[7], 11) + W[19] + 0x7D84A5C3;
	temp[6] = ROTR (F5 (temp[1], temp[4], temp[7], temp[5], temp[3], temp[2], temp[0]), 7) + ROTR (temp[6], 11) + W[ 0] + 0x3B8B5EBE;
	temp[5] = ROTR (F5 (temp[0], temp[3], temp[6], temp[4], temp[2], temp[1], temp[7]), 7) + ROTR (temp[5], 11) + W[12] + 0xE06F75D8;
	temp[4] = ROTR (F5 (temp[7], temp[2], temp[5], temp[3], temp[1], temp[0], temp[6]), 7) + ROTR (temp[4], 11) + W[ 7] + 0x85C12073;
	temp[3] = ROTR (F5 (temp[6], temp[1], temp[4], temp[2], temp[0], temp[7], temp[5]), 7) + ROTR (temp[3], 11) + W[13] + 0x401A449F;
	temp[2] = ROTR (F5 (temp[5], temp[0], temp[3], temp[1], temp[7], temp[6], temp[4]), 7) + ROTR (temp[2], 11) + W[ 8] + 0x56C16AA6;
	temp[1] = ROTR (F5 (temp[4], temp[7], temp[2], temp[0], temp[6], temp[5], temp[3]), 7) + ROTR (temp[1], 11) + W[31] + 0x4ED3AA62;
	temp[0] = ROTR (F5 (temp[3], temp[6], temp[1], temp[7], temp[5], temp[4], temp[2]), 7) + ROTR (temp[0], 11) + W[10] + 0x363F7706;

	temp[7] = ROTR (F5 (temp[2], temp[5], temp[0], temp[6], temp[4], temp[3], temp[1]), 7) + ROTR (temp[7], 11) + W[ 5] + 0x1BFEDF72;
	temp[6] = ROTR (F5 (temp[1], temp[4], temp[7], temp[5], temp[3], temp[2], temp[0]), 7) + ROTR (temp[6], 11) + W[ 9] + 0x429B023D;
	temp[5] = ROTR (F5 (temp[0], temp[3], temp[6], temp[4], temp[2], temp[1], temp[7]), 7) + ROTR (temp[5], 11) + W[14] + 0x37D0D724;
	temp[4] = ROTR (F5 (temp[7], temp[2], temp[5], temp[3], temp[1], temp[0], temp[6]), 7) + ROTR (temp[4], 11) + W[30] + 0xD00A1248;
	temp[3] = ROTR (F5 (temp[6], temp[1], temp[4], temp[2], temp[0], temp[7], temp[5]), 7) + ROTR (temp[3], 11) + W[18] + 0xDB0FEAD3;
	temp[2] = ROTR (F5 (temp[5], temp[0], temp[3], temp[1], temp[7], temp[6], temp[4]), 7) + ROTR (temp[2], 11) + W[ 6] + 0x49F1C09B;
	temp[1] = ROTR (F5 (temp[4], temp[7], temp[2], temp[0], temp[6], temp[5], temp[3]), 7) + ROTR (temp[1], 11) + W[28] + 0x075372C9;
	temp[0] = ROTR (F5 (temp[3], temp[6], temp[1], temp[7], temp[5], temp[4], temp[2]), 7) + ROTR (temp[0], 11) + W[24] + 0x80991B7B;

	md[7] += temp[7] = ROTR (F5 (temp[2], temp[5], temp[0], temp[6], temp[4], temp[3], temp[1]), 7) + ROTR (temp[7], 11) + W[ 2] + 0x25D479D8;
	md[6] += temp[6] = ROTR (F5 (temp[1], temp[4], temp[7], temp[5], temp[3], temp[2], temp[0]), 7) + ROTR (temp[6], 11) + W[23] + 0xF6E8DEF7;
	md[5] += temp[5] = ROTR (F5 (temp[0], temp[3], temp[6], temp[4], temp[2], temp[1], temp[7]), 7) + ROTR (temp[5], 11) + W[16] + 0xE3FE501A;
	md[4] += temp[4] = ROTR (F5 (temp[7], temp[2], temp[5], temp[3], temp[1], temp[0], temp[6]), 7) + ROTR (temp[4], 11) + W[22] + 0xB6794C3B;
	md[3] += temp[3] = ROTR (F5 (temp[6], temp[1], temp[4], temp[2], temp[0], temp[7], temp[5]), 7) + ROTR (temp[3], 11) + W[ 4] + 0x976CE0BD;
	md[2] += temp[2] = ROTR (F5 (temp[5], temp[0], temp[3], temp[1], temp[7], temp[6], temp[4]), 7) + ROTR (temp[2], 11) + W[ 1] + 0x04C006BA;
	md[1] += temp[1] = ROTR (F5 (temp[4], temp[7], temp[2], temp[0], temp[6], temp[5], temp[3]), 7) + ROTR (temp[1], 11) + W[25] + 0xC1A94FB6;
	md[0] += temp[0] = ROTR (F5 (temp[3], temp[6], temp[1], temp[7], temp[5], temp[4], temp[2]), 7) + ROTR (temp[0], 11) + W[15] + 0x409F60C4;

        for(i=0; i<32; ++i)W[i] = 0;
    }




}