/*----------------------------------------------------------------------------
    test.c	C test program
-----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include <malloc.h>

#include "mystd.h"
#include "message.h"
#include "matvec.h"
#include "qrmat.h"

/*----------------------------------------------------------------------------
    simple IO procedure for TestQRMat
-----------------------------------------------------------------------------*/

#define comment '*'

void Read(FILE* f, char *format, ... ); /* prototype */

void ReadLine(FILE* f, char *s) {

   s[0] = comment; s[1] = '\0';
   do 
     if (!fgets(s, 100, f)) Error("Read: line not readable");
   while (s[0] == comment);
} 
   


/*----------------------------------------------------------------------------
    test QR decomposition
-----------------------------------------------------------------------------*/

#define R  8.31434     /* universal constant */

static void TestQRMat() {
    Int i, j, m = 3, n = 5, q;
    char vecString[200];
    FILE *f;

    RealMat A = NewRealMat(1, m, 1, n);
    RealVec x = NewRealVec(1, n);
    RealVec domainScale = NewRealVec(1, n);
    RealVec b = NewRealVec(1, m);
    RealVec imageScale = NewRealVec(1, m);
    RealMat t;
    QRMat *qrMat = NewQRMat(A, m, n);

    Real c, s, sAct, Ti, Ki;

    Message("very simple test example checking scaling and rank defect handling");
    FillRealVec(1.0, 1, n, domainScale);
    domainScale[5] = 5;
    
    FillRealVec(1.0, 1, m, imageScale);
    imageScale[3] = 6;


    qrMat->domainScale = domainScale;
    qrMat->imageScale = imageScale; 

    A[1][1] = 6;
    A[2][2] = 5;
    A[3][3] = 4;
    sprintfRealMat(vecString,"%.2lg", A, 1, m, 1, n);
    Message(" A = %s",vecString);

    b[1] = 1;

    QRDecompose(qrMat);
    QRSolve(qrMat, b, x);
    sprintfRealVec(vecString, "%.3lg", b, 1, m);
    Message("right hand b = %s", vecString);
    sprintfRealVec(vecString, "%.3lg", x, 1, n);
    Message("rank = %d solution x = %s", qrMat->p, vecString);

    b[1] = 2; b[2] = 1;

    QRSolve(qrMat, b, x);
    sprintfRealVec(vecString, "%.3lg", b, 1, m);
    Message("right hand b = %s", vecString);
    sprintfRealVec(vecString, "%.3lg", x, 1, n);
    Message("rank = %d solution x = %s", qrMat->p, vecString);

    A[2][2] = 0;
    sprintfRealMat(vecString,"%.2lg", A, 1, m, 1, n);
    Message(" now A = %s",vecString);

    QRDecompose(qrMat);
    QRSolve(qrMat, b, x);
    sprintfRealVec(vecString, "%.3lg", b, 1, m);
    Message("right hand b = %s", vecString);
    sprintfRealVec(vecString, "%.3lg", x, 1, n);
    Message("rank = %d solution x = %s", qrMat->p, vecString);

    Message("dimension of kernel = %d", q = (n-qrMat->p));
    t = NewRealMat(1, q , 1, n);
    QRKernelBasis(qrMat, t);
    for (j=1; j<=q; j++) {
	Message("t[%d] = %s", j, PrintRealVec("%lf", t[j], 1, n));
    }

    OrthoProject(t, t[1], domainScale, q, n, x);
    Message("projected vector x = %s", PrintRealVec("%lf", x, 1, n));

    FreeRealMat(A, 1, m, 1, n);
    FreeRealVec(domainScale, 1, n);
    FreeRealVec(imageScale, 1, m);
    FreeRealVec(x, 1, n);
    FreeRealVec(b, 1, m);
    FreeQRMat(qrMat);


/*----------------------------------------------------------------------------
  Exercise 3.3 from the Book:
  P. Deuflhard and A. Hohmann, 'Numerische Mathematik -- Eine
     algorithmisch orientierte Einfuehrung', de Gruyter Verlag,
     Berlin, 1991
-----------------------------------------------------------------------------*/

    Message("Arrhenius Reaction Kinetics least squares problem");
    f = fopen("exercise.3.3","r");
    ReadLine(f, vecString);
    sscanf(vecString,"%d", &m); n = 2;
    qrMat = NewQRMat( A = NewRealMat(1, m, 1, n), m, n);
    b = NewRealVec(1, m);
    x = NewRealVec(1, n);
    
    FillRealVec(1.0, 1, m, qrMat->imageScale = NewRealVec(1, m));
    /* shoud be filled with the dKi */

    for (j = 1; j<=m; j++) {
      ReadLine(f, vecString);
      sscanf(vecString,"%d %le %le", &i, &Ti, &Ki);
      A[i][1] = 1.0;
      A[i][2] = -1.0/(R*Ti);
      b[i]    = log(Ki);
    }
    
    QRDecompose(qrMat);
    Message("Exercise 3.3: Matrix has rank %d and condition %le", qrMat->p, qrMat->cond);

    QRSolve(qrMat, b, x);
    Message(" A = %lg,   E = %lg, residual norm = %lg", exp(x[1]), x[2], QRLastResidual(qrMat));

    FreeRealVec(qrMat->imageScale, 1, m);
    FreeQRMat(qrMat);
    FreeRealMat(A, 1, m, 1, n);
    FreeRealVec(x, 1, n);
    FreeRealVec(b, 1, m);
    fclose(f);


/*----------------------------------------------------------------------------
    Kahans counterexample
    ref.: for example Golub, van Loan: Matrix Computations, 
                      The John Hopkins University Press, Baltimore 1985
                      chapter 6, example 6.4-2.
-----------------------------------------------------------------------------*/

    n = 50; c = 0.2; s = sqrt(1.0-SQR(c));
    A = NewRealMat(1, n, 1, n);
    for (i=1, sAct = 1.0; i<=n; i++, sAct *= s) {
      A[i][i] = sAct;
      for (j=i+1; j<=n;j++) A[i][j] = -c*sAct;
    }
    qrMat = NewQRMat(A, n, n);
    qrMat->condMax = 1e+4;
    Message("Kahans Counterexample with dimension = %d", n);
    QRDecompose(qrMat);
    Message("rank = %d, condition estimate: %lg", qrMat->p, qrMat->cond);

    NewRank(qrMat, 0);
    FreeQRMat(qrMat);
    FreeRealMat(A, 1, n, 1, n);
}


void main() {
    InitMystd();
    InitMessage();
    TestQRMat();
    Message("done");
    exit(0);
}
