/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "kask.h"
#include "kaskcmd.h"

#define SKIPWHITE(tp) while(isspace(*(tp))) tp++;
#define SKIPSPACE(tp) while((*(tp)==' ')||(*(tp)=='\t')) tp++;
#define SKIPDIGITS(tp) while (isdigit(*(tp))) tp++;
#define SKIPTOCLASS(tp,class) while ((stdCmdModes.delimClass)[*(tp)]!=class) tp++;
#define SKIPUPTOCLASS(tp,class) while ((stdCmdModes.delimClass)[*(tp)]<class) tp++;
#define INCLASS(tp,class) ((stdCmdModes.delimClass)[*(tp)]==class)
#define UPTOCLASS(tp,class) ((stdCmdModes.delimClass)[*(tp)]>=class)
#define NOTINCLASS(tp,class) ((stdCmdModes.delimClass)[*(tp)]!=class)
#define NOTUPTOCLASS(tp,class) ((stdCmdModes.delimClass)[*(tp)]<class)

#define endClass 127
#define commandClass 10
#define parameterClass 5
#define nameClass 5
#define quoteClass 1

/* **************************************************************
	Static variables used to define the "command language" of
	the commands definition file.
************************************************************** */

static char *initDefNames[] =
  {
    "cmddelim",
	"pardelim",
	"quote",
	"prompt",
	"maxpars",
	"escape",
	"comment",
	nil
  };
static Names defNames = initDefNames;	
static COMMAND staticCmd;

/* **************************************************************
	Global variables used to predefine a "command language".
************************************************************** */

CMDMODES stdCmdModes =
  {
    true,	/* int ignoreCases	Cases significant? */
	99,		/* int maxPars		Maximum number of parameters */
	"Cmd:",	/* stdPrompt		Prompt */
	nil,	/* TextElem *textStack Stack of text elements */
	10,		/* int maxCmdStack	Stack limit */
	-1,		/* int cmdStackP	Stack pointer */
	-1,		/* int noOfNames	Size of list of command names etc */
	-1,		/* int maxIndex		Size of list of command procedures */
	nil,	/* Names names		List of command names */
	nil,	/* int *index		List of routine numbers */
	nil,	/* CMDPROC *procs	List of command procedures */
	nil,	/* char **short		List of short descriptions */
	nil,	/* Names *Pars		List of parameter name lists */
	nil,	/* CharClass delimClass delimiter class vector */
	&staticCmd,	/* COMMAND *cmd	Current command */
	'\\',	/* char escape		escape character */
	'%'		/* char comment		comment character */
  };

static void BreakPars(), RemoveComment();
static int ReadCmdSettings(), BuildNames(), MakeCharClasses(), ConvertNumber();

/* **************************************************************
	DoCommands
	  stacks a string of command for future processing.
************************************************************** */

int DoCommands(line, proc)
  char *line;
  PROC proc;
  {
	if ((stdCmdModes.cmdStackP)==(stdCmdModes.maxCmdStack))
	  {
	    sprintf(globBuf,"Cmd: too many active do file(s): %d\n",
		        stdCmdModes.cmdStackP);
		ZIBStdOut(globBuf);
		return false;
	  }
	(stdCmdModes.cmdStackP)++;
	(stdCmdModes.textStack)[stdCmdModes.cmdStackP].text = line;
	(stdCmdModes.textStack)[stdCmdModes.cmdStackP].pos = line;
	(stdCmdModes.textStack)[stdCmdModes.cmdStackP].proc = proc;
	return true;
  }

/* **************************************************************
	ExecCommands
	  executes a command.
************************************************************** */

int ExecCommand(cmd)
  COMMAND *cmd;
  {
	if ((stdCmdModes.procs)[cmd->no]==nil)
	  {
	    sprintf(globBuf,"Cmd: no code for '%s'(%d)\n",
		        (cmd->pars)[0], cmd->no);
		ZIBStdOut(globBuf);
		return false;
	  }
	return ((stdCmdModes.procs)[cmd->no])(cmd);
  }

/* **************************************************************
	GetNextCommands
	  extracts a command from the text stack and builds the
	  command data structure.
************************************************************** */

COMMAND* GetNextCommand()
  {
    char *tp, *name, tSave;
	int no;
	TextElem *TE;
	COMMAND *stdCmd = stdCmdModes.cmd;

	if ((stdCmdModes.cmdStackP)==-1)
	  { ZIBStdOut("Cmd: nothing to do\n"); goon = false; return nil; }
	TE = &((stdCmdModes.textStack)[stdCmdModes.cmdStackP]);
	tp = TE->pos;
	RemoveComment(tp);
	while (*tp==NUL)
	  {
	    if ((TE->proc)!=nil) (TE->proc)(TE->text);
		(stdCmdModes.cmdStackP)--;
		if ((stdCmdModes.cmdStackP)==-1) return nil;
		TE = &((stdCmdModes.textStack)[stdCmdModes.cmdStackP]);
		tp = TE->pos;
	  }
	name = tp;
	no = CheckName(&tp, stdCmdModes.names, nameClass);
	if (no==-1)
	  {
		SKIPUPTOCLASS(tp, nameClass);
		tSave =*tp; *tp = NUL;
		if ((*name)!=NUL)
		  {
		    sprintf(globBuf,"Cmd: unknown command '%s'\n", name);
			ZIBStdOut(globBuf);
		  }
		*tp = tSave;
		SKIPUPTOCLASS(tp, commandClass);
		if (*tp!=NUL) tp++;
		SKIPWHITE(tp);
		TE->pos = tp;
		return (COMMAND*)-1L;
	  }
	if (NOTUPTOCLASS(tp, commandClass)) { *tp = NUL; tp++; }
	stdCmd->no = (stdCmdModes.index)[no];
	stdCmd->names = (stdCmdModes.pars)[(stdCmdModes.index)[no]];
	(stdCmd->pars)[0] = name;
	BreakPars(&tp, stdCmd->pars, &(stdCmd->noOfPars));
	SKIPWHITE(tp);
	TE->pos = tp;
	return stdCmd;
  }

/* **************************************************************
	SetCommand
	  links a procedure address to a number.
************************************************************** */

void SetCommand(no,proc)
  int no;
  PROC proc;
  {
    if ((no<0)||(no>=(stdCmdModes.maxIndex)))
	  {
	    sprintf(globBuf, "Cmd: %d not allowed to be used as a procedure index, maximum is %d\n",
		        no, stdCmdModes.maxIndex);
		ZIBStdOut(globBuf);
		return;
	  } 
	(stdCmdModes.procs)[no] = proc;
	return;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	BreakPars
	  breaks 'text' into parameters.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static void BreakPars(text, pars, no)
  char **text;
  char **pars;
  int *no;
  {
    char *tp = *text;
	int k = 1;

	SKIPSPACE(tp);
	if (UPTOCLASS(tp, commandClass))
	  {
		if (*tp!=NUL) { *tp = NUL; tp++; }
	    *text = tp;
		pars[1] = nil;
		*no = 0;
		return;
	  }
	while (*tp!=NUL)
	{
	  SKIPSPACE(tp);
	  if (INCLASS(tp, quoteClass))
	    {
		  pars[k] = ConvertString(&tp, (char*)nil, 0);
		  if (pars[k]==nil)
		    {
			  ZIBStdOut("Cmd: closing string quote missing\n");
			  k++; break;
			}
		  else { k++; if (NOTINCLASS(tp, commandClass)) tp++; }
		}
	  else 
	    {
		  pars[k] = tp; k++;
		  SKIPUPTOCLASS(tp, parameterClass);
		}
	  if (INCLASS(tp, commandClass))
	    { if (*tp!=NUL) { *tp = NUL; tp++;  } break; }
	  else
	    { if (*tp!=NUL) { *tp = NUL; tp++;  } }
	}
	*text = tp;
	if (*(pars[k-1])==NUL)
	  k--;   /* empty parameter at end of command line */
	pars[k] = nil;
	*no = k-1;

	return;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	PrintCmdSettings
	  prints the values of some global variables of the
	  command modul.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void PrintCmdSettings()
  {
    ZIBStdOut("Cmd: current settings of the command interpreter\n");
	sprintf(globBuf,"   Prompt:'%s'\n",stdCmdModes.prompt);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"   Maximal number of parameters: %d\n", stdCmdModes.maxPars);
	ZIBStdOut(globBuf);
	if (stdCmdModes.ignoreCases) ZIBStdOut("   Cases are ignored\n");
	sprintf(globBuf,"   Escape symbol: '%c'\n", stdCmdModes.escape);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"   Comment symbol: '%c'\n",stdCmdModes.comment);
	ZIBStdOut(globBuf);
	return;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	PrintNames
	  prints a list of all available command names with their
	  number and short description.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void PrintNames()
  {
	int k = 0, i;
	Names names;

	ZIBStdOut("Cmd: list of commands:\n");
	while ((stdCmdModes.names)[k]!=nil)
	{
	  sprintf(globBuf, "%-11.10s %3d %8x %s\n", (stdCmdModes.names)[k],
	  			(stdCmdModes.index)[k],
				(stdCmdModes.procs)[(stdCmdModes.index)[k]],
				(stdCmdModes.shortDes)[k]);
	  ZIBStdOut(globBuf);
	  names = (stdCmdModes.pars)[(stdCmdModes.index)[k]];
	  i = 0;
	  if (names!=nil)
		{
	      while ((*names)!=nil)
			{
		  	  if (i==0) ZIBStdOut("      ");
			  else { if ((i%6)==0) ZIBStdOut("\n      "); }
		  	  sprintf(globBuf, "%-11.10s", *names);
			  ZIBStdOut(globBuf);
			  names++;
			  i++;
			}
		  ZIBStdOut("\n");
		}
	  k++;
	}
	return;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	CheckName
	  scans 'text' for the occurence of a name from 'names'
	  and returns the corresponding position or -1.
	  Delimiters are defined by 'charClass'.
	  White space at the beginning is skipped.
	  In case of success 'text' is updated.
	  The mode 'ignorCase' is respected.
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int CheckName(text, names, charClass)
  char **text;
  Names names;
  int charClass;
  {
    int no = 0;
	char *p, *q, *firstNonBlank = *text;

	SKIPSPACE(firstNonBlank);
	while(names[no]!=nil)
	{
	  p = firstNonBlank;
	  q = names[no];
	  while(((stdCmdModes.ignoreCases)?(isupper(*p)?tolower(*p):*p):*p)==*q)
	  {
	    if ((*p==NUL)&&(*q==NUL)) break;
		p++; q++;
	  }
	if ((*q==NUL)&&(UPTOCLASS(p, charClass)))
	  { *text = p; return no; }
	no++;
	}
	return -1;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	ParsCheck
	  checks if the number of parameters is betwqeen min and max.
	  Generates an error message.
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int ParsCheck(cmd, min, max)
  COMMAND *cmd;
  int min, max;
  {
    int no =cmd->noOfPars;

	if (no<min) 
	  {
	    sprintf(globBuf, "Cmd: %s: not enough parameters\n",
		        (cmd->pars)[0]);
		ZIBStdOut(globBuf);
		return true;
	  }
	if (no>max) 
	  {
	    sprintf(globBuf, "Cmd: %s: too many parameters\n",
		        (cmd->pars)[0]);
		ZIBStdOut(globBuf);
		return true;
	  }
	return false;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	RemoveComments
	  looks for the command character and substitute the comment
	  by blanks.
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static void RemoveComment(tp)
  char *tp;
  {
    int commentMode = false;

	while(NOTINCLASS(tp, commandClass)&&NOTINCLASS(tp, endClass))
	{
	  if ((*tp)==(stdCmdModes.comment)) commentMode = true;
	  if (commentMode) *tp = ' ';
	  tp++;
	}
	return;
  }

/* **************************************************************
	InitCommands:
	  reads from the file specified by 'path'&'name' the definition
	  of the available commands. The parameter 'procs' contains
	  an array of corresponding procedures.
	 
	  Result is true or false.

	  Used static procedures: ReadCmdPars, BuildNames.

************************************************************** */

int InitCommands(path, name)
  char *path, *name;
  {
	char *completeFileName = globBuf, *text;
	int k;

	if (!MakeCharClasses()) return false;

	strcpy(completeFileName, path);
	strcat(completeFileName, name);

	if (!ZIBReadFile(completeFileName, &text))
	  { 
	    sprintf(globBuf, "Cmd: can't read init-file '%s'\n",
		        completeFileName);
	    ZIBStdOut(globBuf);
		return false;
	  }
	if(!ReadCmdSettings(&text, defNames))
	  { ZIBStdOut("Cmd: error in reading settings\n"); return false; }
	if (!BuildNames(&text))
	  { ZIBStdOut("Cmd: errors in reading names\n"); return false; }
	(stdCmdModes.cmd)->pars = (char**)ZIBAlloc((long)MAX_PAR*sizeof(char*));
	if (((stdCmdModes.cmd)->pars)==nil)
	  {
	    ZIBStdOut("InitCommand: not enough storage(pars)\n");
	    return false;
	  }
	for (k = 0; k<MAX_PAR; k++) ((stdCmdModes.cmd)->pars)[k] = nil;
	stdCmdModes.textStack =
		(TextElem*)ZIBAlloc((long)(stdCmdModes.maxCmdStack)*sizeof(TextElem));
	if ((stdCmdModes.textStack)==nil)
	  {
	    ZIBStdOut("InitCommand: not enough storage(textStack)\n");
	    return false;
	  }
	return true;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	ReadCmdSettings
	  analyses 'text' until a line not starting with dollar
	  is reached. The commands $Delimiter, $Prompt,
	  and $MaxPars, each on a single line are recognized and
	  the corresponding values in 'stdCmdModes' are set.

	  The pointer 'text' is advanced to the next line. Result may
	  be true or false.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
static int ReadCmdSettings(text, commandNames)
  char **text;
  Names commandNames;
  {
    char *tp = *text, *tpReturn, *name;
	int no;
	CharClass delimClass = stdCmdModes.delimClass;

	while(*tp=='$')
	{
	  tp++;								/* skip dollar */
	  name = tp;
	  no = CheckName(&tp, commandNames, nameClass);
	  switch(no)
	  {
	    case 0:							/* $CmdDelim */
		  tp++;
		  delimClass[*tp] = commandClass;
		  break;
	    case 1:							/* $ParDelim */
		  tp++;
		  delimClass[*tp] = parameterClass;
		  break;
	    case 2:							/* $Quote */
		  tp++;
		  delimClass[*tp] = parameterClass;
		  break;
		case 3:							/* $Prompt */
		  SKIPSPACE(tp);
		  tpReturn = ConvertString(&tp, (char*)nil, 0);
		  if (tpReturn==nil)
		    {
			  ZIBStdOut("Cmd: error reading string (Prompt)\n");
			  return false;
			}
		  stdCmdModes.prompt =tpReturn;
		  break;
		case 4:							/* $MaxPar */
		  SKIPSPACE(tp);
		  if (!ConvertNumber(&tp, &(stdCmdModes.maxPars)))
		    {
			  ZIBStdOut("Cmd: error reading number (MaxPar)\n");
			  return false;
			}
		  break;
		case 5:							/* $Escape */
		  SKIPSPACE(tp);
		  stdCmdModes.escape = *tp;;
		  break;
		case 6:							/* $Comment */
		  SKIPSPACE(tp);
		  stdCmdModes.comment = *tp;;
		  break;
		default:
		  SKIPUPTOCLASS(tp,nameClass);
		  sprintf(globBuf, "Cmd: unknown command '$%s'\n", name);
		  ZIBStdOut(globBuf);
		  SKIPUPTOCLASS(tp,commandClass);
		  return false;
	  }							/* End command switch */
	  SKIPUPTOCLASS(tp,commandClass);
	  SKIPWHITE(tp);
	  *text = tp;
	}							/* End command while */
	return true;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	BuildNames
	  generates from 'text' a Names table using 'delims'.
	  The result may be 'nil'.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
static int BuildNames(text)
  char ** text;
  {
    int maxInd = -1, k = 0, i, no;
	char *tp = *text;
	char *tempCmdNames[MAX_COM];
	int tempCmdIndex[1000];
	char *tempCmdShort[1000];
	Names names;

	SKIPWHITE(tp);
	while ((*tp!=NUL)&&(*tp!='$'))
	{
	  tempCmdNames[k] = tp;
	  SKIPUPTOCLASS(tp, parameterClass);
	  if (*tp!=NUL) { *tp = NUL; tp++; }
	  SKIPSPACE(tp);
	  if (!ConvertNumber(&tp, &(tempCmdIndex[k])))
		{
		  sprintf(globBuf, "Cmd: error reading index for %s\n",
		          tempCmdNames[k]);
		  ZIBStdOut(globBuf);
		  return false;
		}
	  if (tempCmdIndex[k]>maxInd) maxInd = tempCmdIndex[k];
	  SKIPSPACE(tp);
	  tempCmdShort[k] = ConvertString(&tp, (char*)nil, 0);
	  k++;
	  if (k==(MAX_COM-1))
	    {
		  sprintf(globBuf, "Cmd: too many command names, maximum is %d\n", k);
		  ZIBStdOut(globBuf);
		  return false;
		}
	  SKIPWHITE(tp);
	}
	maxInd++;
	stdCmdModes.noOfNames = k;
	stdCmdModes.maxIndex = maxInd;
	stdCmdModes.names = (char**)ZIBAlloc((long)(k+1)*sizeof(Names));
	if ((stdCmdModes.names)==nil)
	  {
	    ZIBStdOut("BuildNames: not enough storage(names)\n");
	    return false;
	  }
	stdCmdModes.index = (int*)ZIBAlloc((long)(k+1)*sizeof(int*));
	if ((stdCmdModes.index)==nil)
	  {
	    ZIBStdOut("BuildNames: not enough storage(index)\n");
	    return false;
	  }

	stdCmdModes.shortDes = (char**)ZIBAlloc((long)(k+1)*sizeof(char**));
	if ((stdCmdModes.shortDes)==nil)
	  {
	    ZIBStdOut("BuildNames: not enough storage(short)\n");
	    return false;
	  }
	stdCmdModes.procs = (CMDPROC*)ZIBAlloc((long)maxInd*sizeof(CMDPROC*));
	if ((stdCmdModes.procs)==nil)
	  {
	    ZIBStdOut("BuildNames: not enough storage(procs)\n");
	    return false;
	  }
	stdCmdModes.pars = (Names*)ZIBAlloc((long)maxInd*sizeof(Names*));
	if ((stdCmdModes.pars)==nil)
	  {
	    ZIBStdOut("BuildNames: not enough storage(pars)\n");
	    return false;
	  }
	for (i=0; i<k; i++)
	  {
	    (stdCmdModes.names)[i] = tempCmdNames[i];
		(stdCmdModes.index)[i] = tempCmdIndex[i];
		(stdCmdModes.shortDes)[i] = tempCmdShort[i];
	  }
	(stdCmdModes.names)[k] = nil;
	(stdCmdModes.index)[k] = nil;
	(stdCmdModes.shortDes)[k] = nil;
	for (i=0; i<maxInd; i++)
	  {
	    (stdCmdModes.procs)[i] = nil;
	    (stdCmdModes.pars)[i] = nil;
	  }
	while (*tp=='$')
	{
	  tp++;
	  if (!ConvertNumber(&tp, &no))
	    {
		  ZIBStdOut("Cmd: missing number for parameter keyword list\n");
		  return false;
		}

	  if ((no<0)||(no>maxInd))
	    {
		  sprintf(globBuf, "Cmd: %d not allowed to be used as a procedure index, maximum is %d\n", no, maxInd);
		  ZIBStdOut(globBuf);
		  return false;
		}

	  k = 0;
	  while ((*tp!=NUL)&&(*tp!='$'))
	  {
	    SKIPWHITE(tp);
		tempCmdNames[k] = tp;
		SKIPUPTOCLASS(tp, nameClass);
		if (*tp!=NUL) { *tp = NUL; tp++; }
		k++;
	  }
	  names = (char**) ZIBAlloc((long)(k+1)*sizeof(char*));
	  if (names==nil)
	    {
	      ZIBStdOut("BuildNames: not enough storage(pars)\n");
	      return false;
	    }
	  for (i=0; i<k; i++) names[i] = tempCmdNames[i];
	  names[k] = nil;
	  (stdCmdModes.pars)[no] = names;
	  SKIPWHITE(tp);
	}
	*text = tp;
	return true;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	MakeCharClass
	  allocates (if neccessary) and computes the CharClass-fields
	  in a delimiter data structure.
	  Returns false if not enough space.
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static int MakeCharClasses()
  {
    int k;

	if ((stdCmdModes.delimClass)==nil)
	  stdCmdModes.delimClass = (char*)ZIBAlloc(256L);
	if ((stdCmdModes.delimClass)==nil) return false;
	for (k=0; k<256; k++) (stdCmdModes.delimClass)[k] = 0;
	(stdCmdModes.delimClass)[NUL]	= endClass;
	(stdCmdModes.delimClass)['\n']	= commandClass;
	(stdCmdModes.delimClass)[';']	= commandClass;
	(stdCmdModes.delimClass)[' ']	= parameterClass;
	(stdCmdModes.delimClass)['\t']	= commandClass;
	(stdCmdModes.delimClass)['\'']	= quoteClass;
	(stdCmdModes.delimClass)['\"']	= quoteClass;
	return true;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	ConvertNumber
	  converts a decimal number from the 'text' input. If
	  successful, 'text' is updated.
	  Returns true or false.
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static int ConvertNumber(text, no)
  char **text;
  int *no;
  {
    int k, rc;
	char *tp = *text;

	rc = sscanf(tp,"%d",&k);
	if (rc==1)
	  {
	    *no=k;
		SKIPSPACE(tp);
		if (((*tp)=='+')||((*tp)=='-')) tp++;
		SKIPDIGITS(tp);
		*text = tp;
		return true;
	  }
	return true;
  }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	ConvertString
	  converts a string, evaluates escape sequences.
	  If 'target`==nil the string is converted in place, starting
	  behind the first quote character. If successful,
	  'text' is updated.
	  Returns false if not enough space.
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

char *ConvertString(text, target, max)
  char **text, *target;
  int max;
  {
    char *sp = *text, *tp;
	char quote, escape = stdCmdModes.escape;

	SKIPTOCLASS(sp,quoteClass);
	quote = *sp;
	sp++;
	if (target==nil) { target = sp; max = 99999; }
	tp = target;

	while (*sp!=quote)
	{
	  if (*sp!=escape) *tp = *sp;
	  else
	    {
		  sp++;
		  if (INCLASS(sp,commandClass)) return nil;
		  switch (*sp)
		  {
		    case 'n': *tp = '\n'; break;
			case 't': *tp = '\t'; break;
			case '0': *tp = '0'; break;
			default:  *tp = *sp; break;
		  }
		}
	  sp++; tp++; max--;
	  if (max<0) return nil;
	}
	*tp = NUL;
	*text = sp+1;
	return target;
  }

/* **************************************************************
	CmdQuit
	  set 'goon' false to finish main event loop.
************************************************************** */

int CmdQuit(cmd)
  COMMAND *cmd;
  {
    if (ParsCheck(cmd,0,0)) return false;
	goon = false;
	return true;
  }

/* **************************************************************
	CmdPrint
	  print settings and command names.
************************************************************** */

int CmdPrint(cmd)
  COMMAND *cmd;
  {
    char *tp;
	int no;

	if (ParsCheck(cmd,0,1)) return false;
	if ((cmd->noOfPars)==0)
	  { PrintCmdSettings(); PrintNames(); }
	else
	  {
	    tp = (cmd->pars)[1];
		no =CheckName(&tp, cmd->names, nameClass);
		switch (no)
		{
		  case 0:	PrintNames(); break;
		  case 1:	PrintCmdSettings(); break;
		  default:
		    sprintf(globBuf, "\Cmd: unknown parameter name '%s' for '%s'n",
			        (cmd->pars)[1], (cmd->pars)[0]);
			ZIBStdOut(globBuf);
			break;
		}
	  }
	return true;
  }

/* **************************************************************
	CmdDo
	  reads files and stacks them for processing.
************************************************************** */

int CmdDo(cmd)
  COMMAND *cmd;
  {
    char *text;

    if (ParsCheck(cmd,1,1)) return false;
	if (!ZIBReadFile((cmd->pars)[1],&text)) return false;
	return DoCommands(text,(PROC)ZIBFree);
  }

/* **************************************************************
	CmdExit
	  skips commands of a text stack entry.
************************************************************** */

int CmdExit(cmd)
  COMMAND *cmd;
  {
    if (ParsCheck(cmd,0,0)) return false;
	return true;
  }

/* **************************************************************
	CmdHelp
	  print short description description or list of  names.
************************************************************** */

int CmdHelp(cmd)
  COMMAND *cmd;
  {
    char *tp;
	Names names = stdCmdModes.names;
	int no, i = 0;

	if (ParsCheck(cmd,0,1)) return false;
	if ((cmd->noOfPars)==0)
	  {
	  	ZIBStdOut("Help: list of commands:\n");
		while ((*names)!=nil)
		  {
			if (i==0) ZIBStdOut("      ");
			else { if ((i%6)==0) ZIBStdOut("\n      "); }
			sprintf(globBuf, "%-11.10s", *names);
			ZIBStdOut(globBuf);
			names++;
			i++;
		  }
		ZIBStdOut("\n");
	  }
	else
	  {
	    tp = (cmd->pars)[1];
		no =CheckName(&tp, stdCmdModes.names, nameClass);
		if (no<0)
		  {
		    sprintf(globBuf, "Help: unknown command '%s'\n",
			        (cmd->pars)[1]);
			ZIBStdOut(globBuf);
		  }
		else
		  {
			sprintf(globBuf, "Help: %-11.10s: '%s'; parameters:\n",
			        (stdCmdModes.names)[no],
	  			    (stdCmdModes.shortDes)[no]);
			ZIBStdOut(globBuf);
			names = (stdCmdModes.pars)[(stdCmdModes.index)[no]];
			i = 0;
			if (names!=nil)
			  {
	  			while ((*names)!=nil)
				  {
		  			if (i==0)  ZIBStdOut("      ");
					else { if ((i%6)==0) ZIBStdOut("\n      "); }
					sprintf(globBuf, "%-11.10s", *names);
					ZIBStdOut(globBuf);
					names++;
					i++;
				  }
			ZIBStdOut("\n");
			  }
		  }
	  }
	return true;
  }
