/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/

#include <stdio.h>
#include <math.h>

#include "kask.h"
#include "kaskcmd.h"
#include "kasktri.h"
#include "kaskass.h"
#include "kasksol.h"

SOLVE *actSolve = nil;

REAL mbar, Theta, newFactor = 1.0;
REAL discrErr, discrMaxErr, discrTstErr;
REAL eNormU = ZERO, eNormUUl=ZERO;
REAL totalTime=ZERO, solTime=ZERO, estiTime=ZERO, refTime=ZERO;
REAL iteTime=ZERO, dirTime=ZERO, assTime=ZERO, numTime=ZERO, drawTime;
REAL ares, res;


static int CompDiscrErrP(p)
  PT *p;
  {
    REAL adiff, diff;
	REAL *solVals = (actProblem->solVals);
	int class = p->class;

	(actProblem->Sol)(p->x,p->y,class,solVals);
	diff = RA(p,R_SOL)-solVals[0];
	discrErr += diff*diff;
    adiff = fabs(diff);
	if (discrMaxErr < adiff) 
		discrMaxErr = adiff;
	if (discrTstErr < adiff) 
		discrTstErr = adiff;

	return true;
  }

static int CompDiscrErrE(ed)
  EDG *ed;
  {
    REAL adiff;
	PT pm;
	REAL *solVals = (actProblem->solVals);
	REAL rvec[6];

	pm.vec = &(rvec[0]);
    if ((ed->boundP) != DIRICHLET)	
		{		
		pm.boundP = (ed->boundP);
		(ed->MidPoint)(ed,&pm);
		(actProblem->Sol)(pm.x,pm.y,ed->class,solVals);
		adiff = fabs(RD(pm,R_SOL)-solVals[0]);
		if (discrTstErr < adiff) 
			discrTstErr = adiff;  
		}
	return true;
  }

static int CompDiscrErrT(t)
  TR *t;
  {
    REAL adiff;
	REAL f11,f12,f21,f22;
	REAL xCenter, yCenter, valCenter, xm, ym;
	REAL xPt,yPt,valPt;
	REAL *solVals = (actProblem->solVals);
	PT *p1=t->p1, *p2=t->p2, *p3=t->p3;
	PT pm;
	EDG *ed=t->e1;

	REAL rvec[6];

	pm.vec = &(rvec[0]);
	
	xCenter = 1.0/3.0;
	yCenter = 1.0/3.0;

	f11 = (p2->x) - (p1->x);
	f12 = (p3->x) - (p1->x);
	f21 = (p2->y) - (p1->y);
	f22 = (p3->y) - (p1->y);

	xm = f11*xCenter + f12*yCenter + (p1->x);
	ym = f21*xCenter + f22*yCenter + (p1->y);
 	(actProblem->Sol)(xm,ym,t->class,solVals);

 	pm.boundP = (ed->boundP);
	(ed->MidPoint)(ed,&pm);	

 	valCenter = RA(p1,R_SOL) + 2.0/3.0*(RD(pm,R_SOL)-RA(p1,R_SOL));
    adiff = fabs(valCenter-solVals[0]);
 		if (discrTstErr < adiff) 
			discrTstErr = adiff;  

	xPt = (xm + (p1->x))*HALF;
	yPt = (ym + (p1->y))*HALF;
 	(actProblem->Sol)(xPt,yPt,t->class,solVals);

 	valPt = ( RA(p1,R_SOL) + valCenter)*HALF;
    adiff = fabs(valPt-solVals[0]);
 		if (discrTstErr < adiff) 
			discrTstErr = adiff; 
 
	xPt = (xm + (p2->x))*HALF;
	yPt = (ym + (p2->y))*HALF;
 	(actProblem->Sol)(xPt,yPt,t->class,solVals);

 	valPt = ( RA(p2,R_SOL) + valCenter)*HALF;
    adiff = fabs(valPt-solVals[0]);
 		if (discrTstErr < adiff) 
			discrTstErr = adiff; 
 
	xPt = (xm + (p3->x))*HALF;
	yPt = (ym + (p3->y))*HALF;
 	(actProblem->Sol)(xPt,yPt,t->class,solVals);

 	valPt = ( RA(p3,R_SOL) + valCenter)*HALF;
    adiff = fabs(valPt-solVals[0]);
 		if (discrTstErr < adiff) 
			discrTstErr = adiff; 


	return true;
  }

void Direct(verboseP)
  int verboseP;
  {
	int rc, dim;
	REAL localDrawTime, localDirTime, localAssTime;

	dim = actTriang->noOfPoints;

	localAssTime = ZIBSeconds();
	if (!Assemble())
	  { actSolve->dirSolP = false; actSolve->dirFail = true; return; }
	localAssTime = ZIBSeconds() - localAssTime;
	assTime   += localAssTime;
	totalTime += localAssTime;


	localDirTime = ZIBSeconds();

    if (verboseP)
	  {
	    sprintf(globBuf, "Solve: start of direct cholesky, n=%d\n", dim);
		ZIBStdOut(globBuf);
	  }
    rc = CholDec(actSolve->stiff, actSolve->diagonal, actSolve->decomp,
				 dim);
    if (!rc)
      {
        ZIBStdOut("Solve: direct cholesky failed (CholDec)\n");
        actSolve->dirSolP = false;
		actSolve->dirFail = true;
        return;
      }
    rc = CholSol(actSolve->decomp, actSolve->diagonal, actSolve->rhs,
				 actSolve->solution, dim);
    if (!rc)
      {
        ZIBStdOut("Solve: direct cholesky failed (CholSol)\n");
        actSolve->dirSolP = false;
		actSolve->dirFail = true;
        return;
      }

    rc = ChkSol(actSolve->stiff, actSolve->rhs, actSolve->solution,
				actSolve->diagonal, dim, verboseP);
	if (!rc)
	  {
        ZIBStdOut("Direct: stiffness matrix not positive definite\n");
        actSolve->dirSolP = false;
		actSolve->dirFail = true;
        return;
	  }

    actSolve->dirSolP = true;
    actSolve->dirFail = false;
	actSolve->dirLevel = actTriang->refLevel;
    actSolve->lastStep = DIRECT;
    actSolve->iteEps = ZERO;
	if (verboseP) ZIBStdOut("Solve: end of direct cholesky\n");
	
	StoreVecToField(actSolve->solution,R_SOL);
	StoreVecToField(actSolve->rhs,R_RHS);
	actTriang->lastDirectPoint = actTriang->lastPoint;

	localDirTime = ZIBSeconds()-localDirTime;
	dirTime   += localDirTime;
	totalTime += localDirTime;

	localDrawTime = ZIBSeconds();
	AutomaticRedraw(false, true, false, false);
	localDrawTime = ZIBSeconds()-localDrawTime;
	drawTime += localDrawTime;

    if (verboseP)
       ZIBStdOut("\n step   level depth  dim    estiErr  iter#    iteErr    trueErr on grid         time\n");

		  if ((actProblem->Sol)!=nil)
		    {
			  discrErr  = ZERO;
			  discrMaxErr = ZERO;
			  discrTstErr = ZERO;
		 	  ApplyP(CompDiscrErrP,all);	 
		 /*	  ApplyE(CompDiscrErrE,nodal);	 */
			  ApplyT(CompDiscrErrT,all);	
			  sprintf(globBuf,"Direct    %3d   %3d  %5d                                %9.3e          ",
		          actTriang->refLevel, actTriang->maxDepth,
				  actTriang->noOfPoints, discrMaxErr);
			}
			else
		 	 sprintf(globBuf,"Direct    %3d   %3d  %5d                                                   ",
		          actTriang->refLevel, actTriang->maxDepth,actTriang->noOfPoints);
             
		  ZIBStdOut(globBuf);

          if (verboseP)
             {
             sprintf(globBuf,"\n");
		     ZIBStdOut(globBuf);
             }
	return;
  }

static void CompMeanVals(verboseP)
  int verboseP;
  {
	InitNumAss(N_LQ);
    (actSolve->compNormP) = false;
	if (!axMul(R_SOL, R_RQ))
	  { actSolve->estiFail = true; return; }
	AssRSideDiag();
	mbar = CompRes(R_RES, R_RQ, R_DIAG, R_RHS, &(actSolve->globEps));
	actSolve->globEps += actSolve->iteEps;

	InitNumAss(N_STD);
	if (verboseP)
	  {
		sprintf(globBuf, "Solve: estiEps=%9.3e(%e), mbar=%9.3e (CompEVals)\n",actSolve->globEps,
				sqrt(actSolve->globEps),mbar);
		ZIBStdOut(globBuf);
	  }
	return;
  }

static void CompExtrVals(verboseP)
  int verboseP;
  {
	InitNumAss(N_LQ);
    (actSolve->compNormP) = false;

	if (!axMul(R_SOL, R_RQ))
	   { actSolve->estiFail = true; return; }
	AssRSideDiag();
	mbar = CompResExtrapol(R_RES, R_RQ, R_DIAG, R_RHS, &(actSolve->globEps));
	actSolve->globEps += actSolve->iteEps;

	InitNumAss(N_STD);
	if (verboseP)
	  {
		sprintf(globBuf, "Solve: estiEps=%9.3e(%e), mbar=%9.3e (CompEVals)\n",actSolve->globEps,
				sqrt(actSolve->globEps),mbar);
		ZIBStdOut(globBuf);
	  }
	return;
  }

static void CompExactVals(verboseP)
  int verboseP;
  {    
	EDG *ed = actTriang->firstEdge;
	PT pm;
	REAL *solVals = (actProblem->solVals);
	REAL rvec[6];
    REAL sum=ZERO, res;

	pm.vec = &(rvec[0]);

	while (ed!=nil)
		{
			if ( (ed->firstSon) !=nil) {ed = ed->next; continue; }
			if ( ((ed->boundP)==DIRICHLET) || ((ed->type) ==T_GREEN) )
				{
				res = 0.0;
				}
			else
				{
				(ed->MidPoint)(ed,&pm);
				(actProblem->Sol)(pm.x,pm.y,ed->class,solVals);
				res = fabs(RD(pm,R_SOL)-solVals[0]);
		  		}
			RA(ed,R_RES) = res;
			if ( sum < res ) sum = res;
			ed = ed->next;
	 	}

	actSolve->globEps = sum;
	mbar = sum;

	sprintf(globBuf," globEps,mbar %e %e \n",actSolve->globEps,mbar);
	ZIBStdOut(globBuf);
	actSolve->globEps += actSolve->iteEps;
	if (verboseP)
	  {
		sprintf(globBuf, "Solve: estiEps=%9.3e(%e), mbar=%9.3e (CompEVals)\n",actSolve->globEps,
				sqrt(actSolve->globEps),mbar);
		ZIBStdOut(globBuf);
	  }

	return ;
	
  }


static int RefIfLessTheta(ed)
  EDG *ed;
  {		
	if ((RA(ed, R_RES)>=Theta)) 
	  { 
	    if ((ed->t1)!=nil) RefTr(ed->t1);
	    if ((ed->t2)!=nil) RefTr(ed->t2);
	  }
	return true;
  }

void MeanValue(verboseP)
  int verboseP;
  {
	REAL localEstiTime;	
	REAL eNormUh=ZERO;

	localEstiTime = ZIBSeconds();

/*     energy norm of discret solution for relative error	*/
    (actSolve->compNormP) = true;
	if (actSolve->iluP)
	  if (!ILUAssemble())
	    { actSolve->iteFail = true; return; }

   	InitNumAss(N_STD);
  	if (!(actSolve->MatrixMul)(R_SOL, R_CGH))
	  { actSolve->estiFail = true; return; }
	
	eNormUh = scalprod(R_SOL,R_CGH);
	(actSolve->eNorm) = sqrt(eNormUh);
    (actSolve->compNormP) = false;

/* 	sprintf(globBuf," energy norm of Uh    %e \n", (actSolve->eNorm));
	ZIBStdOut(globBuf);
*/

/*	estimation of error in energy norm	*/
	CompMeanVals(verboseP);
	if ((actSolve->globEps)<ZERO)
	  {
	    ZIBStdOut("Estimate: stiffness matrix maybe not positive definite\n");
		actSolve->estiFail = true;
		return;
	  }
    if (verboseP) 
       ZIBStdOut("\n step   level depth  dim    estiErr  iter#    iteErr    trueErr on grid         time\n");
	sprintf(globBuf,"Estimate  %3d   %3d  %5d  %9.3e\n",
		    	actTriang->refLevel, actTriang->maxDepth,
				actTriang->noOfPoints, sqrt(actSolve->globEps));
	ZIBStdOut(globBuf);
	actSolve->estiP = true;
	actSolve->lastStep = ESTIMATE;
	actSolve->estiFail = false;

	localEstiTime = ZIBSeconds() - localEstiTime;
	estiTime  += localEstiTime;
	totalTime += localEstiTime;

	return;
  }
 
void ExtraPol(verboseP)
  int verboseP;
  {
	REAL localEstiTime;
	REAL eNormUh=ZERO;

	localEstiTime = ZIBSeconds();

/*     energy norm of discret solution for relative error	*/
    (actSolve->compNormP) = true;
	if (actSolve->iluP)
	  if (!ILUAssemble())
	    { actSolve->iteFail = true; return; }

   	InitNumAss(N_STD);
  	if (!(actSolve->MatrixMul)(R_SOL, R_CGH))
	  { actSolve->estiFail = true; return; }
	
	eNormUh = scalprod(R_SOL,R_CGH);
	(actSolve->eNorm) = sqrt(eNormUh);
    (actSolve->compNormP) = false;
/*
 	sprintf(globBuf," energy norm of Uh    %e \n", (actSolve->eNorm));
	ZIBStdOut(globBuf);
*/

/*	estimation of error in energy norm	*/
 

	CompExtrVals(verboseP);
	if ((actSolve->globEps)<ZERO)
	  {
	    ZIBStdOut("Estimate: stiffness matrix maybe not positive definite\n");
		actSolve->estiFail = true;
		return;
	  }
    if (verboseP) 
       ZIBStdOut("\n step   level depth  dim    estiErr  iter#    iteErr    trueErr on grid         time\n");
	sprintf(globBuf,"Estimate  %3d   %3d  %5d  %9.3e\n",
		    	actTriang->refLevel, actTriang->maxDepth,
				actTriang->noOfPoints, sqrt(actSolve->globEps));
	ZIBStdOut(globBuf);
	actSolve->estiP = true;
	actSolve->lastStep = ESTIMATE;
	actSolve->estiFail = false;

	localEstiTime = ZIBSeconds() - localEstiTime;
	estiTime  += localEstiTime;
	totalTime += localEstiTime;

	return;
  }

void ExactErr(verboseP)
  int verboseP;
  {
	REAL localEstiTime;

	localEstiTime = ZIBSeconds();

	CompExactVals(verboseP);
	if ((actSolve->globEps)<ZERO)
	  {
	    ZIBStdOut("Estimate: stiffness matrix maybe not positive definite\n");
		actSolve->estiFail = true;
		return;
	  }
	sprintf(globBuf,"Estimate  %3d   %3d  %5d  %9.3e\n",
		    	actTriang->refLevel, actTriang->maxDepth,
				actTriang->noOfPoints, sqrt(actSolve->globEps));
	ZIBStdOut(globBuf);
	actSolve->estiP = true;
	actSolve->lastStep = ESTIMATE;
	actSolve->estiFail = false;

	localEstiTime = ZIBSeconds() - localEstiTime;
	estiTime  += localEstiTime;
	totalTime += localEstiTime;

	return;
  }

void Refine(verboseP)
  int verboseP;
  {
	int initNoOfPoints = actTriang->noOfPoints, levCount = 0;
	REAL localDrawTime, localRefTime;


	if (!(actSolve->estiP))
	  {
	    ZIBStdOut("Refine: no data from the error estimator\n");
		if ((actSolve->Estimate) != nil)
		  {	actSolve->refFail = true; return; }
	  }
	if (verboseP)
	  {
		sprintf(globBuf, "Solve: start adaptive refinement, sigma=%.2f, s=%.1f, %d points\n",
				actSolve->sigma, actSolve->sFactor,
				initNoOfPoints);
		ZIBStdOut(globBuf);
	  }

	actSolve->refFail = false;
	if ((actSolve->Estimate) == nil)
	  {
		localRefTime = ZIBSeconds();

		OpenRef();
		ApplyT(RefTr,all);
		CloseRef(true);
		sprintf(globBuf,"Refine    %3d   %3d  %5d\n",
		            actTriang->refLevel, actTriang->maxDepth,
				    actTriang->noOfPoints);
		ZIBStdOut(globBuf);

		localRefTime = ZIBSeconds() - localRefTime;
		refTime   += localRefTime;
		totalTime += localRefTime;
	  }
	else
	while (true)
	  {
		localRefTime = ZIBSeconds();

		Theta = (actSolve->sigma)*mbar;
		OpenRef();
		ApplyE(RefIfLessTheta, nodal);
		if (!CloseRef(verboseP))
		  {
		    actSolve->refFail = true;
			if (initNoOfPoints==(actTriang->noOfPoints))
			  ZIBStdOut("Solve: solution identical to zero\n");
			break;
		  }
		levCount++;
		actSolve->estiP = false;
        if(verboseP)
          ZIBStdOut("\n step   level depth  dim    estiErr  iter#    iteErr    trueErr on grid         time\n");
		sprintf(globBuf,"Refine    %3d   %3d  %5d\n",
		            actTriang->refLevel, actTriang->maxDepth,
				    actTriang->noOfPoints);
		ZIBStdOut(globBuf);

		localRefTime = ZIBSeconds() - localRefTime;
		refTime   += localRefTime;
		totalTime += localRefTime;

		if (initNoOfPoints*(actSolve->sFactor)<=(actTriang->noOfPoints)) break;
		if ((actSolve->breakDim)<(actTriang->noOfPoints)) break;
		if ((actSolve->breakLevel)<(actTriang->refLevel)) break;
		if ((actSolve->breakDepth)<(actTriang->maxDepth)) break;
	
		(actSolve->Estimate)(verboseP);
		if ((actSolve->globEps)<ZERO)
		  {
		    ZIBStdOut("Estimate: stiffness matrix maybe not positive definite\n");
			actSolve->estiFail = true;
			return;
		  }
	  }
	if (verboseP)
	  {
	    sprintf(globBuf, "Solve: end adaptive refinement, %d new points, %d level(s)\n",
				(actTriang->noOfPoints)-initNoOfPoints,levCount);
		ZIBStdOut(globBuf);
	  }
	newFactor = initNoOfPoints;
	newFactor = newFactor/(actTriang->noOfPoints);
	actSolve->lastStep = REFINE;
	actSolve->refFail = false;

	localDrawTime = ZIBSeconds();
	AutomaticRedraw(true, false, false, false);
	localDrawTime = ZIBSeconds()-localDrawTime;
	drawTime += localDrawTime;
	return;
  }
 
void Iterate(verboseP)
  int verboseP;
  {
    int rc, iterations = actSolve->maxIteSteps;
    REAL eps =(actSolve->globEps)*(actSolve->iteFactor)*newFactor ;
	REAL localDrawTime, localIteTime, localAssTime;

	localAssTime = ZIBSeconds();
	InitNumAss(N_STD);
    rc = AssRSide();

    if (rc==false)
	  { actSolve->iteFail = true; return; }

	if (actSolve->iluP)
	  if (!ILUAssemble())
	    { actSolve->iteFail = true; return; }



    actSolve->iteSteps = 1;
	rc = CGFirst(actSolve->MatrixMul, eps, &res,
				 actSolve->precondP, actSolve->Precond, verboseP);

	if (verboseP)
	  {
  		sprintf(globBuf, "Solve: first CG step: res=%9.3e(%9.3e)\n", res, sqrt(res));
		ZIBStdOut(globBuf);
	  }

	if ((res<ZERO)||(rc==false))
	  { actSolve->iteFail = true; return; }

	localAssTime = ZIBSeconds() - localAssTime;
	assTime   += localAssTime;
	totalTime += localAssTime;
	

	localIteTime = ZIBSeconds();


    if (res>eps)
      { 
    	rc = CGStep(&iterations, &res, eps);
	    actSolve->iteSteps = iterations;
    	if (verboseP)
		  {
		    sprintf(globBuf, "Solve: after %d iteration steps: res=%e\n",
					iterations, res);
			ZIBStdOut(globBuf);
		  }
	    if (rc==false)
		  { actSolve->iteFail = true; return; }
 	  }

	actSolve->iteFail = (res<eps)?false:true;
	actSolve->lastStep = ITERATE;
	actSolve->iteEps = res;
	localIteTime = ZIBSeconds() - localIteTime;
	iteTime   += localIteTime;
	totalTime += localIteTime;

	localDrawTime = ZIBSeconds();
	AutomaticRedraw(false, true, false, false);
	localDrawTime = ZIBSeconds()-localDrawTime;
	drawTime += localDrawTime;

    if (verboseP)
       ZIBStdOut("\n step   level depth  dim    estiErr  iter#    iteErr    trueErr on grid         time\n");

	if ((actProblem->Sol)!=nil)
		{
			discrErr  = ZERO;
			discrMaxErr = ZERO;
			discrTstErr = ZERO;
			ApplyP(CompDiscrErrP,all);	
	/*		  ApplyE(CompDiscrErrE,nodal);	*/
		 	ApplyT(CompDiscrErrT,all);	 

/*			sprintf(globBuf," %3d   %3d  %5d             %3d   %9.3e  %9.3e   %9.3e  %9.3e\n",
			actTriang->refLevel, actTriang->maxDepth,
			actTriang->noOfPoints, actSolve->iteSteps,
			sqrt(actSolve->iteEps),sqrt(discrErr),discrMaxErr,discrTstErr);
*/

 			sprintf(globBuf,"Iterate   %3d   %3d  %5d             %3d   %9.3e    %9.3e          ",
			actTriang->refLevel, actTriang->maxDepth,
			actTriang->noOfPoints, actSolve->iteSteps,
			sqrt(actSolve->iteEps),discrMaxErr);
		}
			else
		  sprintf(globBuf,"Iterate   %3d   %3d  %5d             %3d   %9.3e                       ",
		          actTriang->refLevel, actTriang->maxDepth,
				  actTriang->noOfPoints, actSolve->iteSteps,
				  sqrt(actSolve->iteEps));
		  ZIBStdOut(globBuf);

          if (verboseP) 
             {
              sprintf(globBuf,"\n");
		      ZIBStdOut(globBuf);
             }
	return;
  }
 
void StatisticsTime()
  {
	sprintf(globBuf," \ntime-statistics: \n");
	ZIBStdOut(globBuf);
	sprintf(globBuf,"process      CPU-time[s]  %% of accumul. time \n\n");
	ZIBStdOut(globBuf);
	sprintf(globBuf,"direct       %8.4f        %6.2f\n",dirTime,dirTime/totalTime*100);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"estimate     %8.4f        %6.2f\n",estiTime,estiTime/totalTime*100);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"refine       %8.4f        %6.2f\n",refTime,refTime/totalTime*100);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"iterate      %8.4f        %6.2f\n",iteTime,iteTime/totalTime*100);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"integrate    %8.4f        %6.2f\n\n",assTime,assTime/totalTime*100);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"accumulated  %8.4f        %6.2f\n\n",totalTime,100.);
	ZIBStdOut(globBuf);
	sprintf(globBuf,"solve-cmd    %8.4f        \n\n",solTime);
	ZIBStdOut(globBuf);

	return;
  }

int CheckPreSets()
  {
    SOLVE *sol = nil;
	
	if (actTriang==nil)
	  { ZIBStdOut("Solve: no triangulation - nothing to solve\n"); return false; }

	if ((actTriang->problem)==nil) actTriang->problem = (PTR)actProblem;
	if (actProblem==nil)
	  { ZIBStdOut("Solve: no problem - nothing to solve\n"); return false; }

	if ((actTriang->solve)==nil)
	  {
	    sol = (SOLVE*) ZIBAlloc((long)sizeof(SOLVE));
		if (sol==nil) return false;
		actTriang->solve = (PTR)sol;
		sol->Direct = Direct;
		sol->Estimate = MeanValue;
		sol->Precond = pcxMul;
        sol->MatrixMul = axMul;
		sol->Iterate = Iterate;
		sol->Refine = Refine;
	  }
	else return true;
	actSolve = (SOLVE*)actTriang->solve;
	actSolve->stiff = nil;
	actSolve->rhs = nil;
	actSolve->decomp = nil;
	actSolve->diagonal = nil;
	actSolve->solution = nil;
	actSolve->dirSolP = false;
	actSolve->dirLevel = 0;
	actSolve->dirDim = 0;
	actSolve->dirFail = false;
	actSolve->estiFail = false;
	actSolve->iteFail = false;
	actSolve->lastStep = START;
	actSolve->maxIteSteps = 100;
	actSolve->precondP = true;
	actSolve->preCondName = "(preconditioned by hierarchical bases)";
	actSolve->estiP = false;
	actSolve->estiPosition = 1;
	actSolve->estiType = 1;
	actSolve->estiTreshold = 0.0;
	actSolve->estiQuadraticP = true;
	actSolve->estiName = "(Estimator: meanvalue)";
	actSolve->eNorm = ZERO;
	actSolve->iluP = false;
	actSolve->compNormP = false;
	actSolve->globEps = ONE;
	actSolve->iteEps = ZERO;
	actSolve->breakEps = 1.0e-1;
	actSolve->breakDim = 1000;
	actSolve->breakLevel = 10;
	actSolve->breakDepth = 10;
	actSolve->breakSteps = 11;
	actSolve->iteFactor = 0.01;
	actSolve->sigma = 0.95;
	actSolve->sFactor = 2.0;
	actSolve->omega = 0.6;
	actSolve->layer = all;
	actSolve->broydenDelta = 10;
	actSolve->renumber = false;

	return true;
  }
 
int SolveDir(cmd)
  COMMAND *cmd;
  {
    if (ParsCheck(cmd, 0, 0)) return false;
	if (!CheckPreSets()) return false;
	(actSolve->Direct)(true);
    return true;
  }

int ErrEsti(cmd)
  COMMAND *cmd;
  {
    if (ParsCheck(cmd, 0, 0)) return false;
	if (!CheckPreSets()) return false;
	if ( (actSolve->Estimate) != nil)
		(actSolve->Estimate)(true);
	
	return true;
  }

int Estimator(cmd)
  COMMAND *cmd;
  {
    int index = 0;
	char *tp;

	if (ParsCheck(cmd,0,1)) return false;

	if ((cmd->noOfPars)==1)
	  {
	    tp = (cmd->pars)[1];
		index = CheckName(&tp, cmd->names, nameClass);
	  }
	if (index==-1)
	  {
	    sprintf(globBuf, "Parameter %s of %s not allowed\n",
				(cmd->pars)[1], (cmd->pars)[0]);
	    ZIBStdOut(globBuf);
		return false;
	  }

	if (!CheckPreSets()) return false;
    switch (index)
	{ 
		case 0: actSolve->Estimate = MeanValue;			/*	std			*/
				actSolve->estiName = "(Estimator: standard)";
				actSolve->estiQuadraticP = true;
				break;
		case 1:	actSolve->Estimate = MeanValue;			/*	MeanValue	*/
				actSolve->estiName = "(Estimator: meanvalue)";
				actSolve->estiQuadraticP = true;
				break;
		case 2: actSolve->Estimate = ExtraPol;			/*	ExtraPol	*/
				actSolve->estiName = "(Estimator: extrapol)";
				actSolve->estiQuadraticP = false;
				actSolve->sigma = 1.0;
				actSolve->sFactor = 1.1;
				break;
		case 3:	actSolve->Estimate = nil;				/*	none		*/
				actSolve->estiName = "(Estimator: none)";
				actSolve->estiQuadraticP = false;
		 		actSolve->estiP = true;	 
				actSolve->iteFactor = 1.0e-6;
				break;
		case 4:	actSolve->Estimate = ExactErr;			/*	exact		*/
				actSolve->estiName = "(Estimator: max-Norm)";
				actSolve->estiQuadraticP = false;
				actSolve->sigma = 1.0;
				actSolve->sFactor = 1.0;
				break;

	 } 
	return true;
  }

int PreCond(cmd)
  COMMAND *cmd;
  {    
	int index = 0;
	char *tp;

	if (ParsCheck(cmd,0,1)) return false;

	if ((cmd->noOfPars)==1)
	  {
	    tp = (cmd->pars)[1];
		index = CheckName(&tp, cmd->names, nameClass);
	  }
	if (index==-1)
	  {
	    sprintf(globBuf, "Parameter %s of %s not allowed\n",
				(cmd->pars)[1], (cmd->pars)[0]);
	    ZIBStdOut(globBuf);
		return false;
	  }
	if (!CheckPreSets()) return false;
    switch (index)
	{ 
		case 0: actSolve->Precond = pcxMul;			/*	std				*/
				actSolve->precondP = true;
				actSolve->preCondName = "(standard preconditioned)";
				break;
		case 1:	actSolve->Precond = pcxMul;			/*	diagonal		*/
				actSolve->precondP = true;
				actSolve->preCondName = "(preconditioned by diagonal)";
				break;
		case 2: actSolve->Precond = pcxMul;			/*	lowertriangle	*/
				actSolve->precondP = true;
				actSolve->preCondName = "(preconditioned by lower triangle)";
				break;
		case 3:	actSolve->Precond = pcxMul;			/*	hierarch. bases	*/
				actSolve->precondP = true;
				actSolve->preCondName = "(preconditioned by hierarch. bases)";
				break;
		case 4:	actSolve->Precond = bpxPcxMul;		/*	bpx				*/
				actSolve->precondP = true;
				actSolve->preCondName = "(preconditioned by bpx)";
				break;
		case 5:	actSolve->Precond = nil;			/*	none			*/
				actSolve->precondP = false;
				actSolve->preCondName = "(not preconditioned)";
				break;

	 } 
	return true;
  }

int RefCmd(cmd)
  COMMAND *cmd;
  {
	int index;
	char *par1 = (cmd->pars)[1];

	if (ParsCheck(cmd,0,1)) return false;
	if (!CheckPreSets()) return false;

	if ((cmd->noOfPars)==0)
	  { (actSolve->Refine)(true); return true; }

	index = CheckName(&par1,cmd->names, nameClass);
	if (index==-1)
	  {
		sprintf(globBuf, "Solve: unknown parameter name '%s' for '%s'\n",
				(cmd->pars)[1], (cmd->pars)[0]);
		ZIBStdOut(globBuf);
		return false;
	  }
	if (!OpenRef()) return false;
	switch (index)
	{
	  case 0:
		if (!ApplyT(RefTr, all)) return false;
		break;
	  case 1:
	    if (!RefTr(actTriang->firstTriangle)) return false;
		break;
	  case 2:
	    if (!RefTr(actTriang->lastTriangle)) return false;
		break;
	}
	if (!CloseRef(true))
	  {
	    actSolve->refFail = true;
		return false;
	  }
	AutomaticRedraw(true, false, false, false);
	actSolve->globEps = ONE;
	actSolve->lastStep = REFINE;
	actSolve->refFail = false;

	return true;
  }

int SolveIte(cmd)
  COMMAND *cmd;
  {
    if (ParsCheck(cmd, 0, 0)) return false;
	if (!CheckPreSets()) return false;
    (actSolve->Iterate)(true);
	return true;
  }

static int LinIntPol(ed)
  EDG *ed;
  {
	PT *P1 = ed->p1,*P2 = ed->p2;
	REAL *dirVals = actProblem->dirVals;
	REAL x,y,fMean;

	fMean = 0.5*(RA(ed->p1,R_SOL)+RA(ed->p2,R_SOL));
	if( (ed->boundP) == DIRICHLET)
		{
			x = (P1->x)+((P2->x)-(P1->x))*HALF;
			y = (P1->y)+((P2->y)-(P1->y))*HALF;
			(actProblem->DirichF)(x,y,ed->class,dirVals);
 			RA(ed,R_SOL) = dirVals[0] - fMean;	 
		}
		else
  		  RA(ed,R_SOL) = fMean;	

    return true;
  }

static int LinEmbed(ed)
  EDG *ed;
  {
  	RA(ed,R_SOL) = ZERO;	
    return true;
  }


int Quadratic(cmd)
  COMMAND *cmd;
  {
    REAL res, eps=1.0e-20;
    int iterations=200, rc;
	REAL eNormUl=ZERO, eNormUq=ZERO, eNormUql=ZERO, errNorm=ZERO;

    if (ParsCheck(cmd, 0, 0)) return false;
	if (!CheckPreSets()) return false;

/*     energy norm of linear solution 		*/
    (actSolve->compNormP) = true;
	if (actSolve->iluP)
	  if (!ILUAssemble())
	    { actSolve->iteFail = true; return false; }

   	InitNumAss(N_STD);
  	if (!(actSolve->MatrixMul)(R_SOL, R_CGH))
	  { actSolve->estiFail = true; return false; }
	
	eNormUl = scalprod(R_SOL,R_CGH);
 	sprintf(globBuf, " energy norm of Ul        %e \n", sqrt(fabs(eNormUl)));
	ZIBStdOut(globBuf);
    (actSolve->compNormP) = false;


/* 	  energy norm of error if exact solution is known  	*/

 	if( (actProblem->Sol) != nil )
		{
		InitNumAss(N_EL);
		eNormU   = ZERO;
		eNormUUl = ZERO;
		ApplyT(eNormInt,all);
	 	sprintf(globBuf, " energy norm of U         %e \n", sqrt(fabs(eNormU)));
		ZIBStdOut(globBuf);
		errNorm = eNormUl + eNormU -2.0*eNormUUl; 
 		sprintf(globBuf, " energy norm of U-Ul      %e \n", sqrt(fabs(errNorm)) );
		ZIBStdOut(globBuf);

   		InitNumAss(N_STD);
		}


/*	error in energy norm of (linear solution - quadratic solution)		*/

/*	Set ZERO in the midpoints of the edges							*/
    ApplyE(LinEmbed,nodal);
	
/*	InitNumAss(N_QUAD);   */
	InitNumAss(N_EQ);

/*	storage of the last (linear) solution in all points					*/
	assign(R_SOLSAVE,R_SOL);
	
/*	Computing of the quadratic solution									*/
    rc = AssRSide();
    if (rc==false)
	  { ZIBStdOut("Quadratic assembling failed\n"); return false; }

    ApplyE(LinIntPol,nodal);

    (actSolve->compNormP) = false;

	rc = CGFirst(axMul, eps, &res,
				  false, (VOIDPROC)nil, false);

	sprintf(globBuf, "Solve: first CG step: res=%9.3e(%9.3e)\n", res, sqrt(res));
	ZIBStdOut( globBuf);

	if ((res<ZERO)||(rc==false))
	  { ZIBStdOut("Quadratic cg failed\n"); return false; }

    if (res>eps)
      { 
    	rc = CGStep(&iterations, &res, eps);
    	sprintf(globBuf, "Solve: after %d iteration steps: res=%e\n",
				iterations, res);
		ZIBStdOut( globBuf);
	    if (rc==false)
		  { ZIBStdOut("Quadratic cg failed\n"); return false; }
 	  }


	if ((actProblem->Sol)!=nil)
		{
			discrErr  	= ZERO;
			discrMaxErr = ZERO;
			ApplyP(CompDiscrErrP,all);	
 	        sprintf(globBuf, " max error of Uq on grid   %e \n", discrMaxErr);
		}
	ZIBStdOut( globBuf);

/*     energy norm of quadratic solution 		*/
    (actSolve->compNormP) = true;
  	if (!(axMul)(R_SOL, R_CGH))
	  { actSolve->estiFail = true; return false; }
	
	eNormUq = scalprod(R_SOL,R_CGH);
 	sprintf(globBuf, " energy norm of Uq        %e \n", sqrt(fabs(eNormUq)));
	ZIBStdOut(globBuf);
    (actSolve->compNormP) = false;


/*	energy norm of (Ul - Uq)							*/
    (actSolve->compNormP) = true;

	sub(R_SOLSAVE,R_SOL,R_SOLSAVE);
  	if (!(axMul)(R_SOLSAVE, R_CGH))
	  { actSolve->estiFail = true; return false; }
	
	eNormUql = scalprod(R_SOLSAVE,R_CGH);
 	sprintf(globBuf, " energy norm of Uq-Ul     %e \n", sqrt(fabs(eNormUql)));
	ZIBStdOut( globBuf);
    (actSolve->compNormP) = false;

	InitNumAss(N_STD);

	return true;
  }

int Statistics(cmd)
  COMMAND *cmd;
  {
    if (ParsCheck(cmd, 0, 0)) return false;
	if (!CheckPreSets()) return false;
    StatisticsTime();
	return true;
  }

int SetBreak(cmd)
  COMMAND *cmd;
  {
    int index = 0, no, k, rc;
	double deps;
	REAL eps;
	char *tp;

	if (ParsCheck(cmd, 0, 2)) return false;
	if (!CheckPreSets()) return false;

	if ((cmd->noOfPars)==0)
	  {
		ZIBStdOut("Solve: state of solution process\n");
		sprintf(globBuf, "  level     %5d - %5d\n",
				actTriang->refLevel, actSolve->breakLevel);
		ZIBStdOut( globBuf);
		sprintf(globBuf, "  depth     %5d - %5d\n",
				actTriang->maxDepth, actSolve->breakDepth);
		ZIBStdOut( globBuf);
		sprintf(globBuf, "  dimension %5d - %5d\n",
				actTriang->noOfPoints, actSolve->breakDim);
		ZIBStdOut( globBuf);
		sprintf(globBuf, "  global error %9.3e - %9.3e\n", sqrt(actSolve->globEps),
				sqrt(actSolve->breakEps));
		ZIBStdOut( globBuf);
		sprintf(globBuf, "  max steps for Iterate: %d\n  iteration factor %9.3e\n  last iteration error:%9.3e\n",
		actSolve->maxIteSteps, actSolve->iteFactor, sqrt(actSolve->iteEps));
		ZIBStdOut( globBuf);

		if (actSolve->dirSolP)
		  {
		    sprintf(globBuf, "  direct solution for level %d available\n",
					actSolve->dirLevel);
			ZIBStdOut(globBuf);
		  }
		else ZIBStdOut("  no direct solution available\n");

		if (actSolve->dirFail)
		  ZIBStdOut("  Direct failed\n");
		else if (actSolve->estiFail)
		       ZIBStdOut("  Estimate failed\n");
			 else if (actSolve->iteFail)
			        ZIBStdOut("  Iterate failed\n");
			      else ZIBStdOut("  no failures yet\n");
		return true;
	  }

	for (k = 1; k<=(cmd->noOfPars); k++)
	  {
		tp = (cmd->pars)[k];
		index = CheckName(&tp, cmd->names, nameClass);
		if (index==-1)
		  {
		    sprintf(globBuf, "Solve: '%s' unknown parameter to %s\n",
					(cmd->pars)[k], (cmd->pars)[0]);
			ZIBStdOut(globBuf);
			return false;
		  }
		rc = 0;
		switch (index)
		{
		  case 0:
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->breakDepth = no;
				  break;
		  case 1:
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->breakLevel = no;
				  break;
		  case 2:
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->breakDim = no;
				  break;
		  case 3:
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->breakSteps = no;
				  break;
		  case 4:									/* globalEps */
		          k++;
				  if (k<=(cmd->noOfPars))
					{
				    rc = sscanf((cmd->pars)[k],"%le",&deps);
					eps=deps;
					}
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->breakEps = eps*eps;
				  break;
		  case 5:
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->maxIteSteps = no;
				  break;
		  case 6:
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->precondP = (no!=0)?true:false;
				  break;
		  case 7:									/* iteFactor */
		          k++;
				  if (k<=(cmd->noOfPars))
					{
				    rc = sscanf((cmd->pars)[k],"%le",&deps);
					eps = deps;
					}
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->iteFactor = eps;
				  break;
		  case 8:									/* sFactor */
		          k++;
				  if (k<=(cmd->noOfPars))
					{
				    rc = sscanf((cmd->pars)[k],"%le",&deps);
					eps = deps;
					}
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->sFactor = eps;
				  break;
		  case 9:									/* sigma */
		          k++;
				  if (k<=(cmd->noOfPars))
					{
				    rc = sscanf((cmd->pars)[k],"%le",&deps);
					eps = deps;
					}
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->sigma = eps;
				  break;
		  case 10:									/* ilu */
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->iluP = (no!=0)?true:false;
                  actSolve->MatrixMul = (no!=0)?ILUaxMul:axMul;
				  break;
		  case 11:									/* omega */
		          k++;
				  if (k<=(cmd->noOfPars))
					{
				    rc = sscanf((cmd->pars)[k],"%le",&deps);
					eps = deps;
					}
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->omega = eps;
				  break;
		  case 12:									/* adjust */
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->adjustP = (no!=0)?true:false;
				  break;
		  case 13:									/* blue */
		          k++;
				  if (k<=(cmd->noOfPars))
				    rc = sscanf((cmd->pars)[k],"%d",&no);
				  if (rc!=1)
				    {
					  sprintf(globBuf, "Solve: '%s' should be followed by a number\n",
					  		  (cmd->pars)[k-1]);
					  ZIBStdOut(globBuf);
					  break;
					}
				  actSolve->blueP = (no!=0)?true:false;
				  break;
		}
	  }
	return true;
  }

static int curSteps, breakReason;
static int BreakCond()
  {
	REAL eNorm = (actSolve->eNorm);
    if (actSolve->dirFail) { breakReason = 5; return true; }
    if (actSolve->estiFail) { breakReason = 6; return true; }
    if (actSolve->iteFail) { breakReason = 7; return true; }
	if (eNorm >= 1.0e-6) 
		{
		if ( (actSolve->breakEps) >= (actSolve->globEps)/(eNorm*eNorm) )
		{ breakReason = 4; return true; }
		}
	else
 		if ( (actSolve->breakEps) >= (actSolve->globEps)/(1.e-12) )
		{ breakReason = 4; return true; }

    if ((actSolve->breakDepth)<=(actTriang->maxDepth))
	  { breakReason = 3; return true; }
    if ((actSolve->breakLevel)<=(actTriang->refLevel))
	  { breakReason = 2; return true; }
    if ((actSolve->breakDim)<=(actTriang->noOfPoints))
	  { breakReason = 1; return true; }
	if ((actSolve->breakSteps)<=curSteps)
	  { breakReason = 0; return true; }
    breakReason = -1;
	return false;
  }


int Solve(cmd)
  COMMAND *cmd;
  {
    int saveMaxSteps, index, lastNoOfPoints;
	char *tp, s[256];
	REAL lastTime, accTime = 0.0, localSolTime, localRefTime;

	localSolTime = ZIBSeconds();

	if (ParsCheck(cmd, 0, 1)) return false;
	if (!CheckPreSets()) return false;
    	saveMaxSteps = actSolve->breakSteps;

	if ((cmd->noOfPars)==1)
	  {
		tp = (cmd->pars)[1];
		index = CheckName(&tp, cmd->names, nameClass);
		if (index==-1)
		  {
		    sprintf(globBuf, "Solve: '%s' unknown parameter to %s\n",
					(cmd->pars)[1], (cmd->pars)[0]);
			ZIBStdOut(globBuf);
			return false;
		  }
	    actSolve->breakSteps = 1;
	  }

   ZIBStdOut("\n step   level depth  dim    estiErr  iter#    iteErr    trueErr on grid         time\n");

	curSteps = 0;


	if ((actSolve->lastStep)==START)
	  {
	    lastTime = ZIBSeconds();
		(actSolve->Direct)(false);
		lastTime = ZIBSeconds()-lastTime;
		accTime += lastTime;
		curSteps++;
		sprintf(globBuf," %6.2f\n\n", 100.0*accTime/(actTriang->noOfPoints));
		ZIBStdOut(globBuf);
	  }


	while (!BreakCond())
	{
	  if ((actSolve->lastStep)!=ESTIMATE)
	    {
	      lastTime = ZIBSeconds();
		  drawTime = 0.0;
		  lastNoOfPoints = actTriang->noOfPoints;
		  if ((actSolve->Estimate) == nil)
	 		 {
				localRefTime = ZIBSeconds();
				OpenRef();
				ApplyT(RefTr,all);
				CloseRef(true);
				(actSolve->lastStep) = REFINE;
				localRefTime  = ZIBSeconds() - localRefTime;
				refTime   += localRefTime;
				totalTime += localRefTime;
		 		sprintf(globBuf,"Refine    %3d   %3d  %5d  \n",
		            actTriang->refLevel, actTriang->maxDepth,
				    actTriang->noOfPoints);
				ZIBStdOut(globBuf);
				AutomaticRedraw(true,false,false,false);
		     }
	      else
			  {
		 	   	(actSolve->Estimate)(false);
		  		if (BreakCond()) break;
			    (actSolve->Refine)(false);
			  }
		}
	  if (!(actSolve->estiFail))
	    {
		if ((actSolve->lastStep)!=REFINE)
			(actSolve->Refine)(false);
		(actSolve->Iterate)(false);
	    lastTime = ZIBSeconds()-lastTime-drawTime;
		accTime += lastTime;
		lastNoOfPoints = (actTriang->noOfPoints)-lastNoOfPoints;
        sprintf(globBuf," %6.2f\n\n", 100.0*accTime/(actTriang->noOfPoints));
		ZIBStdOut(globBuf);
		}
	  curSteps++;
	}
	actSolve->breakSteps = saveMaxSteps;
	switch (breakReason)
	{
	case 0:
	  ZIBStdOut("BreakReason: number of steps reached\n");
	  break;
	case 1:
	  ZIBStdOut("BreakReason: number of points reached\n");
	  break;
	case 2:
	  ZIBStdOut("BreakReason: refinement level reached\n");
	  break;
	case 3:
	  ZIBStdOut("BreakReason: depth reached\n");
	  break;
	case 4:
	  ZIBStdOut("BreakReason: global (estimated) error reached\n");
	  break;
	case 5:
	  ZIBStdOut("BreakReason: direct solver failed\n");
	  break;
	case 6:
	  ZIBStdOut("BreakReason: error estimator failed\n");
	  break;
	case 7:
	  ZIBStdOut("BreakReason: iterative solver failed\n");
	  break;
	default:
	  sprintf(globBuf, "Unknown BreakReason: %d\n", breakReason);
	  ZIBStdOut(globBuf);
	  break;
	}

	localSolTime = ZIBSeconds() - localSolTime;
	solTime += localSolTime;
	return true;
  }
