      SUBROUTINE BVPSOL(FCN,BC,IVPSOL,N,M,T,X,EPS,ITMAX,INFO,
     *NONLIN,IRW,RW,IIW,IW,NI2W,I2W)
C*    Begin Prologue BVPSOL
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL FCN,BC,IVPSOL
      INTEGER N,M
      DOUBLE PRECISION T(M)
      DOUBLE PRECISION X(N,M)
      DOUBLE PRECISION EPS
      INTEGER ITMAX
      INTEGER INFO
      INTEGER NONLIN,IRW
      DOUBLE PRECISION RW(IRW)
      INTEGER IIW
      INTEGER IW(IIW)
      INTEGER NI2W
      INTEGER I2W(NI2W)
C
C     ------------------------------------------------------------
C
C*  Title
C
C     (B)oundary (V)alue (P)roblem (So)lver for highly nonlinear
C     two point boundary value problems using a (L)ocal linear
C     solver (condensing algorithm) for the solution of the arising 
C     linear subproblems.
C
C*  Written by        P. Deuflhard, G.Bader, L. Weimann
C*  Purpose           Solution of nonlinear two-point boundary value
C                     problems.
C*  Method            Local Nonlinear two-point Boundary Value
C                     Problems solver (Multiple shooting approach)
C*  Category          I1b2a - Differential and integral equations
C                             Two point boundary value problems
C*  Keywords          Nonlinear boundary value problems, Multiple
C                     shooting, Newton methods
C*  Version           1.1
C*  Revision          January 1991
C*  Latest Change     January 1991
C*  Library           CodeLib
C*  Code              Fortran 77, Double Precision
C*  Environment       Standard Fortran 77 environment on PC's,
C                     workstations and hosts.
C*  Copyright     (c) Konrad Zuse Zentrum fuer
C                     Informationstechnik Berlin
C                     Heilbronner Str. 10, D-1000 Berlin 31
C                     phone 0049+30+89604-0,
C                     telefax 0049+30+89604-125
C*  Contact           Lutz Weimann
C                     ZIB, Numerical Software Development
C                     phone: 0049+30+89604-185 ;
C                     e-mail:
C                     RFC822 notation: weimann@sc.zib-berlin.de
C                     X.400: C=de;A=dbp;P=zib-berlin;OU=sc;S=Weimann
C
C*    References:
C
C     /1/ R.Bulirsch:
C         Die Mehrzielmethode zur numerischen Loesung von
C         nichtlinearen Randwertproblemen und Aufgaben der
C         optimalen Steuerung.
C         Carl-Cranz-Gesellschaft: Tech.Rep. (Oct.1971)
C
C     /2/ J.Stoer, R.Bulirsch:
C         Einfuehrung in die Numerische Mathematik II.
C         Berlin, Heidelberg, New York: Springer (1st Ed. 1973)
C
C     /3/ P.Deuflhard:
C         A Modified Newton Method for the Solution of
C         Ill-Conditioned Systems of Nonlinear Equations with
C         Application to Multiple Shooting.
C         Numer. Math. 22, 289-315 (1974)
C
C     /4/ P.Deuflhard:
C         Recent Advances in Multiple Shooting Techniques.
C         (Survey Article including further References)
C         In: I.Gladwell, D.K.Sayers (Ed.): Computational
C         Techniques for Ordinary Differential Equations.
C         Section 10, P.217-272.
C         London, New York: Academic Press (1980)
C
C     /5/ P.Deuflhard, G.Bader:
C         Multiple Shooting Techniques Revisited.
C         Univ. Heidelberg, SFB 123, Tech. Rep. 163 (1982)
C
C     /6/ P. Deuflhard:
C         Newton Techniques for Highly Nonlinear Problems -
C         Theory, Algorithms, Codes.
C         Academic press, to appear.
C
C  ---------------------------------------------------------------
C
C* Licence
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time.
C    In any case you should not deliver this code without a special
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C* Warranty
C    This code has been tested up to a certain level. Defects and
C    weaknesses, which may be included in the code, do not establish
C    any warranties by ZIB. ZIB does not take over any liabilities
C    which may follow from aquisition or application of this code.
C
C* Software status
C    This code is under partial care of ZIB and belongs to ZIB software
C    class 2.
C
C     ------------------------------------------------------------
C
C     External subroutines (to be supplied by the user)
C     =================================================
C
C       FCN(N,T,Y,DY)         Right-hand side of system of
C                             first-order differential equations
C         N                   Input: Number of first order ODE's
C         T                   Input: Actual position in the
C                             interval  A ,  B 
C         Y(N)                Input: Values at T
C         DY(N)               Output: Derivatives at T
C
C       BC(YA,YB,R)           Two-point boundary conditions at ( A
C                             = T(1),  B = T(M))
C         YA(N)               Input: Values at A = T(1)
C         YB(N)               Input: Values at B = T(M)
C         R(N)                Output: Values of
C                             boundary conditions function
C
C       IVPSOL(N,FCN,T,Y,TEND,TOL,HMAX,H,KFLAG)
C                             Initial value problem (IVP)
C                             integrator
C         N                   Number of first-order ODE's
C         FCN                 Right-hand side of the ODE's system
C                             ( see above )
C         T                   Input: Starting point of integration
C                             T.LT.TEND
C                             Output: Achieved final point of
C                             integration
C         Y(N)                Input and Output: Values at T
C         TEND                Input: Prescribed final point of
C                             integration
C         TOL                 Input: Prescribed relative precision
C                             (>0)
C         HMAX                Input: Maximum permitted stepsize
C         H                   Input: Initial stepsize guess
C                             Output: Stepsize proposal for next
C                             integration step ( H.EQ.0 ,  if
C                             IVPSOL fails to proceed )
C         KFLAG               Input: Print parameter
C                             Output: Error flag ( KFLAG.LT.0
C                             indicates an error ) .
C                             For further details, see IVPSOL .
C
C     Input parameters (* marks inout parameters)
C     ===========================================
C
C       N                     Number of first-order ordinary
C                             differential equations.
C       M                     Number of Shooting nodes.
C                             =2    Single Shooting
C                             >2    Multiple Shooting
C       T(M)                  Single/Multiple Shooting Nodes
C                             ( T(1)= A ,  T(M)= B )
C     * X(N,M)                Start data for Newton iteration.
C       EPS                   Required relative precision of
C                             solution.
C       ITMAX                 Maximum permitted number of
C                             iteration steps.
C     * INFO                  Print parameter:
C                             -1    No print
C                              0    Print initial data, iterative
C                                   values of level functions,
C                                   solution data (or final data,
C                                   respectively)
C                             +1    Additionally print iterates
C                                   T(J),X(I,J),  I = 1,...,N ,  J
C                                   = 1,...,M
C       NONLIN                Boundary value problem
C                             classification by user:
C                             0     Linear boundary value problem.
C                             1     Nonlinear boundary value
C                                   problem. Good initial data
C                                   available.
C                             2     Highly nonlinear boundary
C                                   value problem. Only bad
C                                   initial data available. Small
C                                   initial damping factor in
C                                   Gauss Newton method.
C                             3     Highly nonlinear boundary
C                                   value problem. Only bad
C                                   initial data available. Small
C                                   initial damping factor in
C                                   Gauss Newton method.
C                                   Additionally initial rank
C                                   reduction to separable linear
C                                   boundary conditions.
C
C       NRW                   Dimension of real workspace RW
C                             NRW.GE.N*N*(M+5)+10*M*N+10*N+M
C       RW(NRW)               Real workspace
C
C       NIW                   Dimension of integer workspace IW
C                             NIW.GE.2*N*N+4*N
C       IW(NIW)               Integer workspace
C
C       NI2W                  Dimension of short integer workspace
C                             I2W
C                             ( Dummy, for compatibility with
C                             BVPSOG )
C       I2W(NI2W)             Short integer workspace
C                             (In the actual implementation of the
C                              same type is IW)
C
C
C     Output parameters:
C     ==================
C
C       X(N,M)                Solution data ( or final data,
C                             respectively )
C       INFO                  Information output parameter
C                              >0   Number of iterations performed
C                                   to obtain the solution
C                              <0   BVPSOL termination
C                              -1   Iteration stops at stationary
C                                   point
C                              -2   Iteration stops after ITMAX
C                                   iteration steps ( as indicated
C                                   by input parameter ITMAX )
C                              -3   Integrator failed to complete
C                                   the trajectory
C                              -4   Gauss Newton method failed to
C                                   converge
C                              -5   Given initial values
C                                   inconsistent with separable
C                                   linear boundary conditions
C                              -6   Iterative refinement failed to
C                                   converge
C                              -7   Reliable relative accuracy
C                                   greater than 1.0D-2
C                              -8   Condensing algorithm for
C                                   linear block system fails, use
C                                   global linear solver in
C                                   boundary value problem routine
C                                   BVPSOG
C                             -10   Real or integer work-space
C                                   exhausted
C
C     ------------------------------------------------------------
C
C*    End Prologue
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      INTEGER M1,NM,NM1,NN,NRW,NIW
      DOUBLE PRECISION RELDIF,TOL,XTHR
C:    Begin
C     ------------------------------------------------------------
C     1 Internal parameters
C     Standard values fixed below
C     Scaling threshold
      XTHR = SMALL
C     Prescribed relative precision for numerical integration
      TOL = EPS*1.0D-2
C     Prescribed relative deviation for numerical differentiation
      RELDIF = DSQRT(TOL)
C     Starting value for pseudo - rank of sensitivity matrix E
      IRANK = N
      IF(INFO.GE.0)THEN
C       Print BVPSOL heading lines
1       FORMAT('1',2X,'B V P S O L',2X,5('*'),2X,'V e r s i o n',2
     *  X,'1 . 0',1X,3('*'),//,1X,'Newton',1('-'),'Method ','for ',
     *  'the ','solution ','of ','boundary ','value ','problems',/
     *  /)
        WRITE(6,1)
      ENDIF
C     Initial preparations
      M1 = M-1
      NN = N*N
      NM = N*M
      NM1 = N*M1
C:    WorkSpace: IW
        L4=1
        L5=L4+N
        L6=L5+N
        L7=L6+N
        L8=L7+N
        L9=L8+N*N
        L10=L9+N*N
        NIW=L10-1
C.    End WorkSpace at NIW
C:    WorkSpace: RW
        L11=1
        L12=L11+N*N*M1
        L13=L12+N*N
        L14=L13+N*N
        L15=L14+N*N
        L16=L15+N*N
        L17=L16+N*N
        L18=L17+NM
        L19=L18+NM
        L20=L19+NM
        L21=L20+NM
        L22=L21+NM
        L23=L22+NM
        L24=L23+NM1
        L25=L24+NM1
        L26=L25+NM1
        L27=L26+NM1
        L28=L27+N
        L29=L28+N
        L30=L29+N
        L31=L30+N
        L32=L31+N
        L33=L32+N
        L34=L33+N
        L35=L34+N
        L36=L35+N
        L37=L36+N
        L38=L37+N
        L39=L38+N
        L40=L39+N
        L41=L40+M
        L42=L41+N
        L43=L42+N*N
        NRW=L43-1
C.    End WorkSpace at NRW
C     ------------------------------------------------------------
C     2 Check for sufficient real/integer workspace
2     FORMAT('0','Minimal ','required ','work-space ',':',/,'0',
     *'Real    ','array ','RW(',I4,')',/,'0','Integer ','array ',
     *'IW(',I4,')')
      WRITE(6,2)NRW,NIW
      IF(NRW.LE.IRW.AND.NIW.LE.IIW)THEN
        CALL BVPL(FCN,BC,IVPSOL,N,M,M1,NM,NM1,T,X,EPS,TOL,RELDIF,
     *  NONLIN,IRANK,ITMAX,INFO,XTHR,IW(L4),IW(L5),IW(L6),IW(L7),
     *  IW(L8),IW(L9),RW(L11),RW(L12),RW(L13),RW(L14),RW(L15),RW(
     *  L16),RW(L17),RW(L18),RW(L19),RW(L20),RW(L21),RW(L22),RW(
     *  L23),RW(L24),RW(L25),RW(L26),RW(L27),RW(L28),RW(L29),RW(
     *  L30),RW(L31),RW(L32),RW(L33),RW(L34),RW(L35),RW(L36),RW(
     *  L37),RW(L38),RW(L39),RW(L40),RW(L41),RW(L42))
      ELSE
C       Fail exit work-space exhausted
        IF(INFO.GE.0.AND.NRW.GT.IRW)THEN
3         FORMAT('0','Error: ','real    ','work ','- ','space ',
     *    'exhausted',/)
          WRITE(6,3)
        ENDIF
        IF(INFO.GE.0.AND.NIW.GT.IIW)THEN
4         FORMAT('0','Error: ','integer ','work ','- ','space ',
     *    'exhausted',/)
          WRITE(6,4)
        ENDIF
        INFO = -10
      ENDIF
      RETURN
C     End of driver routine BVPSOL
      END
      SUBROUTINE BVPL(FCN,BC,IVPSOL,N,M,M1,NM,NM1,T,X,EPS,TOL,
     *RELDIF,NONLIN,IRANK,ITMAX,INFO,XTHR,IROW,ICOL,ICOLB,PIVOT,IA,
     *IB,G,A,B,BG,E,QE,DX,DDX,DXQ,DXQA,XA,XW,XU,HH,DHH,HHA,D,DE,R,
     *DR,RA,U,DU,QU,X1,XM,T1,T2,DX1,RF,US,EH)
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL FCN,BC,IVPSOL
      INTEGER N,M,M1,NM,NM1
      DOUBLE PRECISION T(M),X(NM)
      DOUBLE PRECISION EPS
      DOUBLE PRECISION TOL,RELDIF
      INTEGER NONLIN,ITMAX,IRANK
      INTEGER INFO
      DOUBLE PRECISION XTHR
      INTEGER IROW(N),ICOL(N),ICOLB(N),PIVOT(N)
      INTEGER IA(N,N),IB(N,N)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION A(N,N),B(N,N),BG(N,N),E(N,N),QE(N,N)
      DOUBLE PRECISION DX(NM),DDX(NM),DXQ(NM),DXQA(NM),XA(NM),XW(
     *NM),XU(NM1),HH(NM1),DHH(NM1),HHA(NM1),D(N),DE(N),R(N),DR(N),
     *RA(N),U(N),DU(N),QU(N),X1(N),XM(N),T1(N),T2(N),DX1(N),RF(M),
     *US(N)
      DOUBLE PRECISION EH(N,N)
C
C     Additional dimensional integer variables
C     ========================================
C
C       M1                M-1
C       NM                N*M
C       NM1               N*(M-1)
C
C     Internal real arrays (workspace) :
C     ==================================
C
C       G(N,N,M1)        (N,N) -Wronskian Matrices G(1),...,G(M-1)
C                         .
C       A(N,N)            Wronskian Matrix on left boundary
C                         dBC/dX(X(1,...,N),T(1)).
C       B(N,N)            Wronskian Matrix on right boundary
C                         dBC/dX(X((N-1)*M+1,...,N*M),T(M)).
C       BG(N,N)           Workspace for subroutine RHS1. Holds
C                         subsequently the matrices B,B*G(M-1),...,
C                         B*G(M-1)*...*G(2)during computation of
C                         the right hand side of the condensed
C                         linear system.
C       E(N,N)            Sensitivity Matrix of the boundary value
C                         problem: E = A+B*G(M-1)*...*G(1).
C       EH(N,N)           Holds a copy of the row- and column
C                         scaled, but not decomposed sensitivity
C                         matrix E needed for iterative refinement
C                         computations.
C       QE(N,N)           Workspace for DECCON and SOLCON to hold
C                         the updating part of the pseudoinverse
C                         of E in case of it's rank defeciency.
C       D(N)              Diagonal elements of upper triangular
C                         matrix of QR-decomposition computed by
C                         DECCON .
C       DDX(NM)           Workspace for subroutine BLSOLI - gets
C                         the iterative refinement vector of DXQ .
C       DE(N)             Holds row scaling factors for the
C                         sensitivity matrix.
C       DHH(NM1)          Holds the recursive refinement of HH
C                         computed in BLSOLI .
C       DR(N)             Workspace for subroutine BLSOLI to hold
C                         the boundary residual
C                         BC(DXQ(1,...,N),DXQ((M-1)*N+1,...,M*N))+
C                         (A*DXQ(1,...,N))+B*DXQ((M-1)*N+1,...,M*N)
C                         .
C       DU(N)             Workspace for subroutine BLSOLI to hold
C                         the right hand side of the linear system
C                         E*T2 = U-E*DX1 solved to compute the
C                         iterative refinement for DX1 .
C       DX(NM)            Actual newton correction.
C       DXQ(NM)           Simplified Newton correction J(k-1)*X(k)
C                         with the Jacobian J(k) and the iterate
C                         vector X(k) at the k-th iterate.
C       DXQA(NM)          Previous simplified Newton correction
C                         J(k-2)*X(k-1).
C       DX1(N)            Workspace to receive the solution output
C                         of SOLCON within computation of DXQ and
C                         it's iterative refinement.
C       HH(NM1)           Elements (J-1)*N+1 to J*N are holding
C                         the values
C                         Y(T(J+1),X((J-1)*N+1,...,J*N))-X(J*N+1,
C                         ...,(J+1)*N)
C                         ( with the trajectory Y in
C                          T(J),T(J+1) , J = 1,...,M-1 ).
C       HHA(NM1)          Holds the previous values of HH .
C       QU(N)             Savespace to hold U(N)for restoring it
C                         in the case of rank reduction.
C       R(N)              Value of the boundary condition function
C                         BC for the current iterate.
C       RA(N)             Previous values of R .
C       RF(M)             Workspace for subroutine BLSOLI - R(J)
C                          gets the maximum norm of the
C                         components (J-1)*N,...,J*N of the
C                         iterative refinement vector to DXQ for
C                         determination of the sweep index for
C                         subsequent iterations.
C       T1(N)             Workspace used for miscellaneous
C                         purposes temporarely.
C       T2(N)             Workspace used for miscellaneous
C                         purposes temporarely.
C       U(N)              Holds the right hand side of the
C                         condensed linear system computed by
C                         subroutine BLRHS1 .
C       US(N)             Workspace for subroutine BLSOLI to save
C                         U(N).
C       XA(NM)            Previous Newton iterate.
C       XU(NM1)           Elements (J-1)*N+1 to J*N are holding
C                         the values Y(T(J+1),X((J-1)*N+1,...,J*N))
C                         of the trajectory in the interval  T(J),
C                         T(J+1) , (for J = 1,...,M-1 ).
C       XW(NM)            Scaling factors for iteration vector.
C       X1(N)             Components of the iteration vector
C                         corresponding to the left boundary
C                         A = T(1).
C       XM(N)             Components of the iteration vector
C                         corresponding to the right boundary
C                         B = T(M).
C
C     Internal integer arrays (workspace)
C     ===================================
C
C       IROW(N)           Row permutations of sensitivity matrix.
C       ICOL(N)           Column permutations of matrix A
C                         (left boundary).
C       ICOLB(N)          Column permutations of matrix B
C                         (right boundary).
C       PIVOT(N)          Workspace for DECCON and SOLCON to hold
C                         permutations performed during
C                         QR-decomposition.
C       IA(N,N)           Reflects the sparse structure of matrix
C                         A by values 0, 1.
C       IB(N,N)           Reflects the sparse structure of matrix
C                         B by values 0, 1.
C
C     Internal real variables
C     =======================
C
C       COND              Gets the condition of the sensitivity
C                         matrix determined by DECCON . Not
C                         further used.
C       CONDE             Maximum permitted subcondition of
C                         sensitivity matrix E .
C       COND1             Condition of boundary conditions part of
C                         the decomposed sensitivity matrix E .
C       COND2             Condition of the remaining part of the
C                         sensitivity matrix E .
C       CONV              Scaled maximum norm of DXQ computed by
C                         subroutine BLLVLS . Used for convergence
C                         test.
C       CONVA             Holds the previous value of CONV .
C       DEL               Becomes the Euklid's normsquare of the
C                         defect belonging to the condensed linear
C                         system in case of rank defeciency. Used
C                         for computation of damping factor
C                         predictor.
C       EPSMIN            Smallest reasonable permitted accuracy
C                         EPS that can be prescribed by the user.
C       FC                Actual Gauss Newton iteration damping
C                         factor.
C       FCA               Previous Gauss Newton iteration damping
C                         factor.
C       FCDNM             Used to compute the denominator of the
C                         damping factor FC during computation of
C                         it's predictor, corrector and
C                         aposteriori estimate (in the case of
C                         performing a Rank1 update) .
C       FCH               Temporarely used for storing the new FC
C                         when computing aposteriori estimate.
C       FCMIN             Minimum permitted relaxation factor. If
C                         FC becomes smaller than this value, one
C                         of the following may occur:
C                         a.    Recomputation of the sensitivity
C                               matrix by means of difference
C                               approximation (instead of Rank1
C                               update), if Rank1 - update
C                               previously was used
C                         b.    Rank reduction of sensitivity
C                               matrix E ,  if difference
C                               approximation was used previously
C                               and Rank(E).NE.0
C                         c.    Fail exit otherwise
C       FCMINH            DSQRT(FCMIN). Used for rank
C                         determination computations.
C       FCMIN2            FCMIN**2 . Used for FC-predictor
C                         computation.
C       FCNUM             Gets the numerator of the aposteriori
C                         estimate of FC .
C       FCNUMK            Gets the numerator of the corrector
C                         computation of FC .
C       FCNUMP            Gets the numerator of the predictor
C                         computation of FC .
C       H                 Actual integrator stepsize.
C       HMAX              Maximum permitted integrator stepsize.
C                         Set to the length of the integration
C                         interval, e.g. the distance of the
C                         effected Shooting points.
C       HSAVE             Stepsize saved across the call of the
C                         integrator.
C       HSTART            Start stepsize for integration used by
C                         subroutines BLFCNI and BLDERG .
C       MUE               Temporary value used during computation
C                         of damping factors predictor.
C       REDH              Multi purpose reduction factor. (???)
C       RELDIF            Relative deviation for numerical
C                         differentation.
C       SENS1             Sensitivity of boundary conditions part
C                         of the decomposed sensitivity matrix E .
C       SENS2             Sensitivity of the remaining part of the
C                         matrix E .
C       SIGMA             Decision parameter for Jacobian Rank1
C                         updates (SIGMA.GT.1) . Rank1 updates are
C                         inhibited, if SIGMA.GT.1/FCMIN is set.
C       SKAP              Used to compute and print out the
C                         incompatibility factor of the nonlinear
C                         boundary value (e.g. least squares)
C                         problem.
C       SUMF              Standard level of the current iterate,
C                         e.g. Norm2(F(X))**2
C                         with the nonlinear model function F on
C                         which Newton iteration is performed,
C                         arising from the Multiple Shooting
C                         approach.
C       SUMX              Natural level of the current iterate,
C                         e.g. Norm2(DX)
C                         with the Newton correcture DX
C                         (see above).
C       SUMXA             Natural level of the previous iterate.
C       TFAIL             Used to get and print out in case of an
C                         integrator failure the last reached T
C                         value as a proposal for insertion of a
C                         new Shooting point.
C       TOL               Prescribed relative precision for
C                         numerical integration.
C       TOLH              Temporary used for computation of TOL
C                         (may be obmitted|).
C       TOLMIN            Lower bound value for TOL .
C       XTHR              Threshold for scaling.
C       TJ                Used by BLFCNI to hold T(J).
C       TJ1               Used by BLFCNI to hold T(J+1).
C       CORR              Used by BLSOLI to compute RF(J),  J = 1,
C                         ...,M .
C       EPX1              Used by BLSOLI to get the maximum norm
C                         of DX1 corrected by iterative
C                         refinement.
C       EPDX1             Used by BLSOLI to get the maximum norm
C                         of the iterative refinement correcture
C                         for DX1 .
C       EPX1H             Relative accuracy of DX1 after iterative
C                         refinement (= EPDX1/EPX1 ). If EPX1H.GT.
C                         1/2 ,  refinement is considered to be
C                         failed.
C       EPH               Used by BLSOLI as tolerance for the
C                         iterative refinement vectors maximum
C                         norm. If it exceeds at index J the
C                         tolerance, refinement will be performed
C                         next time for the components associated
C                         to Shooting points T(J),...,T(M).
C       SIGDEL            Used by BLSOLI to compute the required
C                         integrator accuracy from the multiple
C                         shooting condition.
C       SIGDLH            Used by BLSOLI to compute the multiple
C                         shooting condition
C                         Max( RF(J+1)/RF(J) ) ,  ( J = 1,...,M-1
C                         ). See above for RF .
C
C     Internal integer variables
C     ==========================
C
C       IC                Permutated index. Used by BLSOLI .
C       ICA               Temporarely used during search for
C                         separable linear boundary conditions at
C                         T(1).
C       ICB               Temporarely used during search for
C                         separable linear boundary conditions at
C                         T(M).
C       IH                Temporarely used during search for
C                         separable linear boundary conditions at
C                         T(1) and T(M).
C       IR                Temporary storage for a row permutation
C                         index.
C       IRANK             Rank of sensitivity matrix E of current
C                         iteration step.
C       IRANKA            Rank of sensitivity matrix E of previous
C                         iteration step.
C       IRANKB            Rank of the decomposed sensitivity
C                         matrix part belonging to the boundary
C                         conditions.
C       IREPET            Parameter of subroutines DECCON and
C                         SOLCON indicating the mode of operation
C                         of these routines:
C                         >=0   Full QR-decomposition required
C                          <0   Rank reduction required
C       IRKMAX            Holds the maximum applied rank of all
C                         previous iterates. Must be necessary
C                         .EQ. NE for convergence.
C       IS                Additional DO loop index.
C       ISUM              Used for determination of sparse
C                         structure of matrices A and B as
C                         nonzeros location counter.
C       ITER              Iteration count.
C       JA                Previous sweep index. Used in subroutine
C                         BLSOLI .
C       JIN               Current sweep index. Used in subroutine
C                         BLSOLI .
C       JJ                Used as "reverse DO loop" index:
C                         JJ = IUPB-J in a loop like DO J = 1,IUPB
C                         ...
C       JN                New sweep index determined in subroutine
C                         BLSOLI .
C       JRED              Damping factor reduction count during an
C                         iterate.
C       KC                Temporary storage for a column
C                         permutation index.
C       KFLAG             Gets the subintervall number of the
C                         failure from subroutine BLDERG
C                         if the integrator failed.
C       KOUNT             Trajectory evaluations count.
C       KPRINT            Print parameter - copy of input
C                         parameter INFO .
C       LEVEL             Flow control parameter needed by
C                         subroutine BLSOLI :
C                         0     indicates computation of Gauss
C                               Newton correcture,
C                         1     indicates computation of
C                               simplified Gauss Newton correcture
C                               (after computation of the
C                               preliminary new iterate)
C       NB                Number of separable boundary conditions
C                         at T(M)
C       NE                Number of not separable boundary
C                         conditions at T(1) (and number of rows
C                         and columns of sensitivity matrix)
C       NEW               Count of subsequent performed Rank1
C                         (Broyden) updates.
C       NY                Iterative refinement sweep count for an
C                         iterate ( used in subroutine BLSOLI ) .
C       NYMAX             Highest allowed iterative refinement
C                         sweep index.
C:    End Parameter
C:    EPMACH = relative machine precision
      DOUBLE PRECISION EPMACH
      PARAMETER (EPMACH=2.23D-16)
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION REDH
      PARAMETER (REDH=1.0D-2)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION TWO
      PARAMETER (TWO=2.0D0)
      DOUBLE PRECISION EIGHT
      PARAMETER (EIGHT=8.0D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=1.0D1)
      DOUBLE PRECISION FCMIN
      PARAMETER (FCMIN=1.0D-2)
      DOUBLE PRECISION FCMIN2
      PARAMETER (FCMIN2=1.0D-4)
      DOUBLE PRECISION FCNLIN
      PARAMETER (FCNLIN=1.0D-2)
      DOUBLE PRECISION SIGMA
      PARAMETER (SIGMA=2.0D0)
      INTEGER I,ICA,ICB,IH,IR,IRANKA,IRANKB,IREPET,IRKMAX,IS,
     *ISUM,ITER,I0,J,JJ,JN,JRED,J1,K,KC,KFLAG,
     *KOUNT,KPRINT,L,LEVEL,NB,NE,NEW,NY,NYMAX
      DOUBLE PRECISION COND,COND1,COND2,CONV,CONVA,DEL,EPH,
     *EPSMIN,FC,FCA,FCDNM,FCH,FCMINH,FCNMP2,FCNUM,FCNUMK,FCNUMP,
     *HSTART,MUE,S,SENS1,SENS2,SIGDEL,SIGDLH,SKAP,
     *SMALIN,ST,SUMF,SUMX,SUMXA,TFAIL,TH,TOLH,TOLMIN,
     *EPX1H,CONDE
      LOGICAL DIFAPP,FCOMPT,JACRFR,NEXT,REDUCT
      INTEGER L1,L2
      DOUBLE PRECISION S1
C:    Begin
C:    Begin of Segment Bvpsol.Body
C       ----------------------------------------------------------
C       1 Initialization
C       ----------------------------------------------------------
C       1.1 Internal parameters
C       Standard values fixed below
C       Minimum relative precision of integrator ( to be adapted )
        TOLMIN = EPMACH*TEN*TEN
C       Maximum permitted number of iterative refinements sweeps
C       Maximum permitted sub - condition number of senitivity
C         matrix E
        CONDE = ONE/EPMACH
        NYMAX = M-1
C
        FCMINH = DSQRT(FCMIN)
C       ----------------------------------------------------------
C       1.1.1 Common parameters
C       Starting value of relaxation factor (FCMIN.LE.FC.LE.1.0)
        IF(NONLIN.LE.1)THEN
C         for linear or mildly nonlinear problems
          FC = ONE
        ELSE
C         for highly nonlinear problems
          FC = FCNLIN
        ENDIF
C       Starting value for pseudo - rank of matrix A
        IRANK = N
C       Minimum reasonable value for EPS
        EPSMIN = DSQRT(TEN*EPMACH)
        IF(EPS.LT.EPSMIN) EPS = EPSMIN
C       ----------------------------------------------------------
C       1.2 Initial preparations
        IF(FC.LT.FCMIN) FC = FCMIN
        IF(FC.GT.ONE) FC = ONE
        KPRINT = INFO
        ITER = 0
        KOUNT = 0
        IREPET = 0
        INFO = -1000
        FCA = FC
        CONV = ONE
        JACRFR = .FALSE.
C:      Begin SetVec.Vec
        DO 5 L1=1,NM
          XA(L1)=X(L1)
5       CONTINUE
C.      End SetVec.Vec
        IF(TOL.LE.ZERO) TOL = EPS/TEN
        IF(TOL.LT.TOLMIN) TOL = TOLMIN
        DIFAPP = .TRUE.
        HSTART =(T(2)-T(1))*REDH
        SENS1 = ZERO
        COND1 = ONE
        SENS2 = ZERO
        COND2 = ONE
        IRKMAX = 0
        IRANKB = 0
        SIGDLH = ZERO
        SUMF = ONE
C:      Mat IA = Scalar (Rows 1,N ; Cols 1,N)
        L1 = 0
        DO 6 L2=1,N
        DO 6 L44=1,N
          IA(L2,L44)=L1
6       CONTINUE
C.      End SetIntMat.S
C:      Mat IB = Scalar (Rows 1,N ; Cols 1,N)
        L1 = 0
        DO 7 L2=1,N
        DO 7 L44=1,N
          IB(L2,L44)=L1
7       CONTINUE
C.      End SetIntMat.S
C:      CubeMat G (layer 1)= Scalar (Rows 1,N ; Cols 1,N)
        S1 = ZERO
        DO 8 L1=1,N
        DO 8 L2=1,N
          G(L1,L2,1)=S1
8       CONTINUE
C.      End SetCubeMat.S
        IF(KPRINT.GE.0)THEN
C         Print Start vector data, predescribed precision and max
C         iteration steps
9         FORMAT('0','Initial ','data',//)
          WRITE(6,9)
          DO 10 J=1,M
11          FORMAT(D13.5,2X)
            WRITE(6,11)T(J)
12          FORMAT((14X,3(D20.10,1X)))
            WRITE(6,12)(X(L1),L1=(J-1)*N+1,J*N)
10        CONTINUE
13        FORMAT('0','N ','=',I2,2X,'M ','=',I2,/,'0',
     *    'Prescribed ','relative ','precision',D10.2,2X,/,'0',
     *    'Maximum ','permitted ','number ','of ','iteration ',
     *    'steps',1X,I3,//,'1')
          WRITE(6,13)N,M,EPS,ITMAX
          IF(KPRINT.EQ.0)THEN
14          FORMAT('0',1X,66('*'))
            WRITE(6,14)
15          FORMAT('0',4X,'It',4X,'Ny',7X,'Levelf',10X,'Levelx',8X,
     *      'Rel.Fc.')
            WRITE(6,15)
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       1.3 Startup step
C       ----------------------------------------------------------
C       1.3.1 Computation of the residual vector
        CALL BLFCNI(IVPSOL,FCN,BC,N,M,NM,NM1,ITER,KPRINT,HSTART,
     *  FCMIN,T,X,X1,XM,T1,XU,HH,R,TOL,FC,FCOMPT,REDUCT,KFLAG,
     *  KOUNT,INFO)
C
C       Main iteration loop
C       ===================
C
C:      While (expression)
16      IF(INFO.EQ.-1000)THEN
C:          Begin of Segment Bvpsol.Core
C             ----------------------------------------------------
C             2 Startup of iteration step
              IF(.NOT.JACRFR)THEN
                LEVEL = 0
C               --------------------------------------------------
C               2.1 Scaling of variables X(NM)
                CALL BLSCLE(N,M,NM,NM1,X,XU,XW,XTHR)
                IF(ITER.NE.0)THEN
                  IRANKA = IRANK
C:                Begin SetVec.Vec
                  DO 17 L1=1,NM
                    DXQA(L1)=DXQ(L1)
17                CONTINUE
C.                End SetVec.Vec
C:                FCNUM = Sum of Formula Elements (for 1,NM)
                  FCNUM = 0.0D0
                  DO 18 L1=1,NM
                    FCNUM=FCNUM+((DX(L1)/XW(L1))**2)
18                CONTINUE
C.                End MakeSum.Comp
C:                FCNMP2 = Sum of Formula Elements (for 1,NM)
                  FCNMP2 = 0.0D0
                  DO 19 L1=1,NM
                    FCNMP2=FCNMP2+((DXQ(L1)/XW(L1))**2)
19                CONTINUE
C.                End MakeSum.Comp
                  FCNUMP = FCNUM*FCNMP2
                ENDIF
                IF(ITER.NE.0)THEN
                  IF(IRANK.GE.NB.AND.FC.GT.FCMINH) IRANK = NE
                  IF(IRANK.LT.NB) IRANK = NB
                  TH = FC-ONE
C:                FCDNM = Sum of Formula Elements (for 1,NM)
                  FCDNM = 0.0D0
                  DO 20 L1=1,NM
                    FCDNM=FCDNM+(((DXQ(L1)+TH*DX(L1))/XW(L1))**2)
20                CONTINUE
C.                End MakeSum.Comp
                  FCH = DSQRT(FCNUM/FCDNM)*FC*FC*HALF
C                 ------------------------------------------------
C                 2.1.1 Decision criterion for Jacobian updating
C                       technique:
C                       DIFAPP.EQ..TRUE. numerical
C                       differentiation,
C                       DIFAPP.EQ..FALSE. rank1 updating
                  DIFAPP = FC.LT.FCA.AND.NEW.GT.0.OR.FCH.LT.FC*
     *            SIGMA.OR.IRANKA.LT.IRANK.OR.EPH*REDH.GT.EPS
                  FCA = FC
                  IF(NONLIN.GT.0) FC = DMIN1(FCH,ONE)
                ENDIF
C               --------------------------------------------------
C               2.2 Difference approximation of jacobian matrix A
C                   ( If Difapp.EQ..TRUE. ) or
C                   Rank-1 update of jacobian matrix A ( If Difapp
C                   .EQ..FALSE. )
                CALL BLDERA(BC,N,M,NM,XW,X1,XM,R,T2,A,B,RELDIF)
C               --------------------------------------------------
C               2.3 Determination of sparse structure of matrices
C                   A and B and determination of internal row
C                   scaling of sensitivity matrix E
                ISUM = 0
                DO 21 I=1,N
                  S = ZERO
                  DO 22 K=1,N
                    TH = DABS(A(I,K))*XW(K)
                    IF(S.LT.TH) S = TH
                    TH = DABS(B(I,K))*XW(K+NM1)
                    IF(S.LT.TH) S = TH
22                CONTINUE
                  IF(S.LT.XTHR) S = XTHR
                  DE(I)=SMALL/S
                  DO 23 K=1,N
                    IF(IA(I,K).LE.0)THEN
                      IF(A(I,K).NE.ZERO)THEN
                        IA(I,K)=1
                        ISUM = 1
                      ENDIF
                    ENDIF
                    IF(IB(I,K).LE.0)THEN
                      IF(B(I,K).NE.ZERO)THEN
                        IB(I,K)=1
                        ISUM = 1
                      ENDIF
                    ENDIF
23                CONTINUE
21              CONTINUE
                IF(ISUM.NE.0)THEN
C                 ------------------------------------------------
C                 2.3.1 Determination of row and column
C                       permutation vectors
                  DO 24 I=1,N
                    ICOL(I)=I
                    ICOLB(I)=I
                    IROW(I)=I
24                CONTINUE
C                 ------------------------------------------------
C                 2.3.2 Search for separable linear boundary
C                       conditions at T(1)
                  NE = N
                  DO 25 I=1,N
                      DO 26 K=1,N
                        IF(IB(I,K).NE.0) GOTO 9996
26                    CONTINUE
                      ISUM = 0
                      DO 27 K=1,N
                        IF(IA(I,K).NE.0)THEN
                          ISUM = ISUM+1
                          ICA = K
                        ENDIF
27                    CONTINUE
                      IF(ISUM.LE.1)THEN
                        DO 28 IS=1,N
                          IH = ICOL(IS)
                          IF(IH.EQ.ICA) ICOL(IS)=ICOL(NE)
                          IH = IROW(IS)
                          IF(IH.EQ.I) IROW(IS)=IROW(NE)
28                      CONTINUE
                        ICOL(NE)=ICA
                        IROW(NE)=I
                        NE = NE-1
                        IF(DABS(R(I)).GT.TEN*EPMACH*DABS(X(ICA)))
     *                  THEN
                          INFO = -5
                          GOTO 9998
                        ENDIF
                      ENDIF
9996                CONTINUE
25                CONTINUE
                  IF(KPRINT.GE.0.AND.NE.EQ.0)THEN
29                  FORMAT('0','Warning: ','attempt ','to ',
     *              'solve ','initial ','value ','problem')
                    WRITE(6,29)
                  ENDIF
                  IF(IRANK.GT.NE) IRANK = NE
                  IRANKA = IRANK
C                 ------------------------------------------------
C                 2.3.3 Search for separable linear boundary
C                       conditions at T(M)
                  NB = 0
                ENDIF
                IF(ISUM.NE.0.AND.NE.NE.0)THEN
                  DO 30 I=1,NE
                      IR = IROW(I)
                      DO 31 K=1,N
                        IF(IA(IR,K).NE.0) GOTO 9995
31                    CONTINUE
                      ISUM = 0
                      DO 32 K=1,N
                        IF(IB(IR,K).NE.0)THEN
                          ISUM = ISUM+1
                          ICB = K
                        ENDIF
32                    CONTINUE
                      IF(ISUM.LE.1)THEN
                        NB = NB+1
                        DO 33 IS=1,N
                          IH = ICOLB(IS)
                          IF(IH.EQ.ICB) ICOLB(IS)=ICOLB(NB)
33                      CONTINUE
                        ICOLB(NB)=ICB
                        IROW(I)=IROW(NB)
                        IROW(NB)=IR
                        IF(DABS(R(IR)).GT.TEN*EPMACH*DABS(X(ICB+
     *                  NM1)))THEN
                          INFO = -5
                          GOTO 9998
                        ENDIF
                      ENDIF
9995                CONTINUE
30                CONTINUE
                  IF(KPRINT.GE.0.AND.NB.EQ.N)THEN
34                  FORMAT('0','Warning: ','attempt ','to ',
     *              'solve ','initial ','value ','problem')
                    WRITE(6,34)
                  ENDIF
C                 Initial rank strategy for highly nonlinear
C                   problems
                  IF(NB.LT.NE.AND.ITER.EQ.0.AND.NONLIN.GT.2) IRANK
     *            = NB
                ENDIF
              ENDIF
              JACRFR = .FALSE.
              IF(DIFAPP)THEN
                NEW = 0
                KFLAG = 0
                CALL BLDERG(FCN,N,NE,M,M1,NM,NM1,T,X,XU,XW,T2,
     *          TFAIL,G,ICOL,IVPSOL,HSTART,TOL,RELDIF,KFLAG)
                IF(KFLAG.LT.0)THEN
                  INFO = -3
                  GOTO 9998
                ENDIF
                IF(M.GT.2) KOUNT = KOUNT+N
                IF(M.EQ.2) KOUNT = KOUNT+NE
              ELSE
                NEW = NEW+1
                CALL BLRK1G(N,M,M1,NM,NM1,XW,DX,HH,HHA,T1,G,FCA)
              ENDIF
C             ----------------------------------------------------
C             2.3.4 Computation of sensitivity matrix E =-A+B*G(M1
C                   *..*G(1))
C                   (projections included)
              IF(IRANK.NE.0)THEN
                DO 35 I=1,NE
                  IR = IROW(I)
C:                Mat E(Row I)= Mat B(Row IR)* Scalar  (for 1,N)
                  S1 = DE(IR)
                  DO 36 L1=1,N
                    E(I,L1)=B(IR,L1)*S1
36                CONTINUE
C.                End SetRow.RowxS
35              CONTINUE
                DO 37 JJ=1,M1
                  J = M-JJ
                  DO 38 I=1,NE
                    DO 39 K=1,N
                      S = ZERO
                      DO 40 L=1,N
                        S = S+E(I,L)*G(L,K,J)
40                    CONTINUE
                      T1(K)=S
39                  CONTINUE
                    DO 41 K=1,N
                      E(I,K)=T1(K)
41                  CONTINUE
38                CONTINUE
37              CONTINUE
C               --------------------------------------------------
C               2.4 Prepare solution of the linear system
C               --------------------------------------------------
C               2.4.1 internal row and column scaling of matrix A
C               INTERNAL ROW AND COLUMN SCALING AND PERMUTATION OF
C                 MATRIX E
                DO 42 K=1,NE
                  KC = ICOL(K)
                  S = XW(KC)
                  DO 43 I=1,NE
                    IR = IROW(I)
                    E(I,K)=-(A(IR,KC)*DE(IR)+E(I,KC))*S
43                CONTINUE
42              CONTINUE
C               --------------------------------------------------
C               2.4.2 Save matrix E on EH
C:              Mat EH = Mat E (Rows 1,NE ; Cols 1,NE)
                DO 44 L1=1,NE
                DO 44 L2=1,NE
                  EH(L1,L2)=E(L1,L2)
44              CONTINUE
C.              End SetMat.Mat
              ENDIF
              IRANKB = NB
C             ----------------------------------------------------
C             2.4.3 Monitor for actually applied maximum rank
              IF(IRKMAX.LT.IRANK) IRKMAX = IRANK
C             ----------------------------------------------------
C             2.5 Save values of R(N)and HH((M-1)*N)
              IF(IREPET.EQ.0)THEN
C:              Begin SetVec.Vec
                DO 45 L1=1,N
                  RA(L1)=R(L1)
45              CONTINUE
C.              End SetVec.Vec
C:              Begin SetVec.Vec
                DO 46 L1=1,NM1
                  HHA(L1)=HH(L1)
46              CONTINUE
C.              End SetVec.Vec
              ENDIF
              NEXT = .FALSE.
C
C             Pseudo-rank reduction loop
C             ==========================
C
C:            DO (Until)
47            CONTINUE
C               --------------------------------------------------
C               3 Main-part of iteration step
C               --------------------------------------------------
C               3.1 Solution of the linear system
C               --------------------------------------------------
C               3.1.1 Constrained QR-decomposition of ( ( COMMA NE
C                     NE ) ) - matrix E
                COND = CONDE
                IF(IRANK.GT.0) CALL DECCON(E,N,N,IRANKB,NE,NE,
     *          IRANK,COND,D,PIVOT,IREPET,QE,T1)
                IF(NONLIN.EQ.0.AND.IRANK.LT.NE)THEN
                  INFO = -8
                  GOTO 9998
                ENDIF
C               --------------------------------------------------
C               3.1.2 evaluation of subcondition and sensitivity
C                     numbers
                COND1 = ONE
                COND2 = ONE
                SENS1 = ZERO
                SENS2 = ZERO
                IF(IRANKB.NE.0)THEN
                  SENS1 = DABS(D(1))
                  COND1 = SENS1/DABS(D(IRANKB))
                ENDIF
                IF(IRANKB.NE.IRANK)THEN
                  SENS2 = DABS(D(IRANKB+1))
                  COND2 = SENS2/DABS(D(IRANK))
                ENDIF
                IF(FCA.GE.1.0D0.AND.FC.GE.1.0D0.AND.ITER.NE.0)THEN
                  IF(IRANKB.NE.IRANK.AND.SENS2.LT.(EPS/REDH)*SMALL)
     *            IRANK = IRANKB
                  IF(IRANKB.NE.0.AND.SENS1.LT.(EPS/REDH)*SMALL)
     *            IRANK = 0
                ENDIF
C               --------------------------------------------------
C               3.1.3 (best) (least squares) solution of linear (N,
C                     N)-system
                CALL BLSOLI(N,M,M1,NM,NM1,LEVEL,NE,NB,IRANK,IRANKB,
     *          IREPET,NYMAX,KPRINT,EPS,REDH,TOLMIN,TOL,RELDIF,EPH,
     *          EPX1H,SIGDEL,SIGDLH,E,EH,HH,DHH,R,A,B,BG,G,QE,U,QU,
     *          DE,DU,T1,T2,US,DX1,D,DDX,DXQ,XW,DR,RF,IROW,ICOL,
     *          ICOLB,PIVOT,NY,INFO)
                IF(INFO.NE.-1000) GOTO 9998
C               --------------------------------------------------
C               3.2 Evaluation of scaled natural level function
C                   SUMX
C                   scaled maximum error norm CONV
C                   evaluation of (scaled) standard level function
C                   SUMF ( SUMF only, if KPRINT.GE.0 )
C                   and computation of ordinary newton corrections
C                   DX(N)
                CALL BLLVLS(N,M,NM,NM1,XW,DXQ,HH,R,DE,CONV,SUMX,
     *          SUMF,KPRINT)
C:              Begin SetVec.Vec
                DO 48 L1=1,NM
                  DX(L1)=DXQ(L1)
48              CONTINUE
C.              End SetVec.Vec
                IF(IREPET.EQ.0.AND.IRANK.NE.0)THEN
C:                Begin SetVec.Vec
                  DO 49 L1=1,IRANK
                    QU(L1)=U(L1)
49                CONTINUE
C.                End SetVec.Vec
                ENDIF
C:              Begin SetVec.Vec
                DO 50 L1=1,NM
                  XA(L1)=X(L1)
50              CONTINUE
C.              End SetVec.Vec
                SUMXA = SUMX
                CONVA = CONV
C               --------------------------------------------------
C               3.3 a - priori estimate of relaxation factor FC
                JRED = 0
                REDUCT = .FALSE.
                IF(ITER.NE.0.AND.NONLIN.NE.0)THEN
                  IF(NEW.LE.0.AND.(IRANK.GE.NE.OR.IRANKA.GE.NE)
     *            .OR.IREPET.NE.0)THEN
C                   ----------------------------------------------
C                   3.3.1 Full rank case (independent of preceding
C                         rank) computation of the denominator of
C                         a-priori estimate
C:                  FCDNM = Sum of Formula Elements (for 1,NM)
                    FCDNM = 0.0D0
                    DO 51 L1=1,NM
                      FCDNM=FCDNM+(((DX(L1)-DXQA(L1))/XW(L1))**2)
51                  CONTINUE
C.                  End MakeSum.Comp
                    IF(IRANK.NE.N)THEN
C                     --------------------------------------------
C                     3.4 Rank - deficient case ( if previous rank
C                         was full )
C                     --------------------------------------------
C                     3.4.1 Computation of the projected
C                           denominator of a-priori estimate
C:                    Vec T1 = Scalar (for 1,N)
                      S1 = ZERO
                      DO 52 L1=1,N
                        T1(L1)=S1
52                    CONTINUE
C.                    End SetVec.S
                      IF(IRANK.NE.0)THEN
                        DO 53 L=1,NE
                          K = ICOL(L)
                          DX1(L)=DXQA(K)/XW(K)
53                      CONTINUE
C                       ------------------------------------------
C                       3.4.2 Projection for reduced component DX1
C                             (NE)
                        CALL BLPRJC(N,NE,IRANK,DEL,DX1,D,T2,QE,
     *                  PIVOT)
                        DO 54 L=1,NE
                          K = ICOL(L)
                          T1(K)=DX1(L)*XW(K)
54                      CONTINUE
                      ENDIF
                      DO 55 J=1,M1
                        DO 56 I=1,N
                          S = ZERO
                          DO 57 K=1,N
                            S = S+T1(K)*G(I,K,J)
57                        CONTINUE
                          T2(I)=S
56                      CONTINUE
C:                      Begin SetVec.Vec
                        DO 58 L1=1,N
                          T1(L1)=T2(L1)
58                      CONTINUE
C.                      End SetVec.Vec
                        I0 = J*N
                        DO 59 I=1,N
                          ST = ONE/XW(I+I0)
                          S = T1(I)
                          DEL = DEL+S*ST*ST*(S+(DX(I+I0)-DXQA(I+I0))
     *                    *TWO)
59                      CONTINUE
55                    CONTINUE
                      FCDNM = FCDNM+DEL
                    ENDIF
                    FCDNM = FCDNM*SUMX
C                   ----------------------------------------------
C                   3.4.3 New relaxation factor
                    IF(FCDNM.GE.FCNUMP*FCMIN2)THEN
                      MUE = FCA*DSQRT(FCNUMP/FCDNM)
                      FC = DMIN1(MUE,ONE)
                    ELSE
                      FC = ONE
                    ENDIF
                  ENDIF
                  IREPET = 0
                  REDUCT = FC.LT.FCMIN
                ENDIF
                LEVEL = 1
                  IF(.NOT.REDUCT)THEN
C                   ----------------------------------------------
C                   3.5 Save natural level for later computations
C                       of corrector and print iterate
                    FCNUMK = SUMX
                    IF(KPRINT.GE.0)THEN
C                     Print Standard - and natural level
                      IF(KPRINT.GT.0)THEN
60                      FORMAT('0',1X,66('*'))
                        WRITE(6,60)
61                      FORMAT('0',4X,'It',4X,'Ny',7X,'Levelf',10X,
     *                  'Levelx',18X,'New',4X,'Rank')
                        WRITE(6,61)
                      ENDIF
62                    FORMAT('0',4X,I2,4X,I2,5X,D10.3,2X,4X,D10.3
     *                ,2X,13X,I2,6X,I2)
                      WRITE(6,62)ITER,NY,SUMF,SUMXA,NEW,IRANK
                      IF(KPRINT.GT.0)THEN
63                      FORMAT('0',1X,66('*'))
                        WRITE(6,63)
                      ENDIF
                    ENDIF
C
C                   Relaxation-factor reduction loop
C                   ================================
C
C:                  DO (Until)
64                  CONTINUE
C                     --------------------------------------------
C                     3.6 Preliminary new iterate
C:                    DO (Until)
65                    CONTINUE
                        FCOMPT = .FALSE.
C:                      Vec X = Vec XA + Vec DX * Scalar (for 1,NM)
                        S1 = FC
                        DO 66 L1=1,NM
                          X(L1)=XA(L1)+DX(L1)*S1
66                      CONTINUE
C.                      End SetVec.Vec&VecxS
                        IF(ITER.GT.ITMAX)THEN
                          INFO = -2
                          GOTO 9997
                        ENDIF
C                       ------------------------------------------
C                       3.6.1 Computation of the residual vector
                        CALL BLFCNI(IVPSOL,FCN,BC,N,M,NM,NM1,ITER,
     *                  KPRINT,HSTART,FCMIN,T,X,X1,XM,T1,XU,HH,R,
     *                  TOL,FC,FCOMPT,REDUCT,KFLAG,KOUNT,INFO)
                        IF(INFO.NE.-1000) GOTO 9997
                        IF(REDUCT) GOTO 9994
                      IF(.NOT.(FCOMPT)) GOTO  65
C.                    UNTIL ( expression - negated above)
C                     --------------------------------------------
C                     3.6.2 (best) (least squares) solution of
C                           linear (N,N) -system
                      CALL BLSOLI(N,M,M1,NM,NM1,LEVEL,NE,NB,IRANK,
     *                IRANKB,IREPET,NYMAX,KPRINT,EPS,REDH,TOLMIN,
     *                TOL,RELDIF,EPH,EPX1H,SIGDEL,SIGDLH,E,EH,HH,
     *                DHH,R,A,B,BG,G,QE,U,QU,DE,DU,T1,T2,US,DX1,D,
     *                DDX,DXQ,XW,DR,RF,IROW,ICOL,ICOLB,PIVOT,NY,
     *                INFO)
                      IF(INFO.NE.-1000) GOTO 9998
C                     --------------------------------------------
C                     3.6.3 Evaluation of scaled natural level
C                           function SUMX
C                           scaled maximum error norm CONV and
C                           evaluation of (scaled) standard level
C                           function SUMF
                      CALL BLLVLS(N,M,NM,NM1,XW,DXQ,HH,R,DE,CONV,
     *                SUMX,SUMF,KPRINT)
C                     --------------------------------------------
C                     3.7 Rank independent convergence test
                      IF(CONV.LE.EPS.AND.IRKMAX.EQ.NE)THEN
                        INFO = 0
                        GOTO 9997
                      ENDIF
C                     --------------------------------------------
C                     3.8 Natural monotonicity test
                      IF(SUMX.GT.SUMXA)THEN
C                       ------------------------------------------
C                       3.9 Output of iterate
                        IF(KPRINT.GE.0)THEN
C                         Print Standard - and natural level, and
C                         damping factor
                          IF(KPRINT.GT.0)THEN
67                          FORMAT('0',1X,66('*'))
                            WRITE(6,67)
68                          FORMAT('0',4X,'It',4X,'Ny',7X,'Levelf',
     *                      10X,'Levelx',8X,'Rel.Fc.')
                            WRITE(6,68)
                          ENDIF
69                        FORMAT('0',4X,I2,4X,I2,5X,D10.3,2X,4X,D10.3
     *                    ,2X,4X,F5.3)
                          WRITE(6,69)ITER,NY,SUMF,SUMX,FC
                          IF(KPRINT.GT.0)THEN
70                          FORMAT('0',1X,66('*'))
                            WRITE(6,70)
                          ENDIF
                        ENDIF
                        JRED = JRED+1
                        IF(NONLIN.EQ.0)THEN
                          INFO = -4
                          GOTO 9997
                        ENDIF
C                       ------------------------------------------
C                       3.10 Compute reduced relaxation factor FC
                        TH = FC-ONE
C:                      FCDNM = Sum of Formula Elements (for 1,NM)
                        FCDNM = 0.0D0
                        DO 71 L1=1,NM
                          FCDNM=FCDNM+(((DXQ(L1)+TH*DX(L1))/XW(L1))
     *                    **2)
71                      CONTINUE
C.                      End MakeSum.Comp
                        FC = DSQRT(FCNUMK/FCDNM)*FC*FC*HALF
C                       Rank reduction, if relaxation factor to
C                         small
                        REDUCT = FC.LT.FCMIN.OR.NEW.GT.0.AND.JRED
     *                  .GT.1
                      ELSE
                        NEXT = .TRUE.
                      ENDIF
                    IF(.NOT.(NEXT.OR.REDUCT)) GOTO  64
C.                  UNTIL ( expression - negated above)
C
C                   End of relaxation-factor reduction loop
C                   =======================================
C
                  ENDIF
9994            CONTINUE
                IF(.NOT.NEXT)THEN
C                 ------------------------------------------------
C                 3.11 Restore former values for repeting
C                      iteration step
                  IREPET = 1
C                 Restore former values
                  LEVEL = 0
C:                Begin SetVec.Vec
                  DO 72 L1=1,N
                    R(L1)=RA(L1)
72                CONTINUE
C.                End SetVec.Vec
C:                Begin SetVec.Vec
                  DO 73 L1=1,N
                    X1(L1)=XA(L1)
73                CONTINUE
C.                End SetVec.Vec
C:                Begin SetVec.Vec
                  DO 74 L1=1,N
                    XM(L1)=XA(L1+NM1)
74                CONTINUE
C.                End SetVec.Vec
C:                Begin SetVec.Vec
                  DO 75 L1=1,NM
                    X(L1)=XA(L1)
75                CONTINUE
C.                End SetVec.Vec
C:                Begin SetVec.Vec&Vec
                  DO 76 L1=1,NM1
                    XU(L1)=X(L1+N)+HHA(L1)
76                CONTINUE
C.                End SetVec.Vec&Vec
C:                Begin SetVec.Vec
                  DO 77 L1=1,NM1
                    HH(L1)=HHA(L1)
77                CONTINUE
C.                End SetVec.Vec
                  IF(KPRINT.GE.0)THEN
78                  FORMAT('0',5X,I2,1X,'Not ','accepted ',
     *              'relaxation ','factor',5X,F5.3,12X,I2)
                    WRITE(6,78)ITER,FC,IRANK
                  ENDIF
                  IF(ITER.EQ.0)THEN
                    FC = FCMIN
                  ENDIF
                  IF(NEW.GT.0)THEN
                    DIFAPP = .TRUE.
                    JACRFR = .TRUE.
                    GOTO 9998
                  ENDIF
C                 ------------------------------------------------
C                 3.12 Pseudo-rank reduction
                  IREPET = -1
                  IF(IRANK.EQ.0)THEN
                    INFO = -3
                    GOTO 9997
                  ENDIF
C:                Begin SetVec.Vec
                  DO 79 L1=1,IRANK
                    U(L1)=QU(L1)
79                CONTINUE
C.                End SetVec.Vec
                  IRANK = IRANK-1
                  IF(IRANKB.GT.IRANK) IRANKB = IRANK
                ENDIF
              IF(.NOT.(NEXT)) GOTO  47
C.            UNTIL ( expression - negated above)
C
C             End of pseudo-rank reduction loop
C             =================================
C
C             ----------------------------------------------------
C             4 Preparations to start the following iteration step
              ITER = ITER+1
              IRANKA = IRANK
C             Preliminary pseudo-rank
              IF(IRANK.GE.NB.AND.FC.GT.FCMINH) IRANK = NE
              IF(IRANK.LT.NB) IRANK = NB
C             ----------------------------------------------------
C             4.1 Print values
              IF(KPRINT.GE.0)THEN
C               Print Standard - and natural level, and damping
C               factor
                IF(KPRINT.GT.0)THEN
80                FORMAT('0',1X,66('*'))
                  WRITE(6,80)
81                FORMAT('0',4X,'It',4X,'Ny',7X,'Levelf',10X,
     *            'Levelx',8X,'Rel.Fc.')
                  WRITE(6,81)
                ENDIF
82              FORMAT('0',4X,I2,4X,I2,5X,D10.3,2X,4X,D10.3,2X,4X,F5.3)
                WRITE(6,82)ITER,NY,SUMF,SUMX,FC
                IF(KPRINT.GT.0)THEN
83                FORMAT('0',1X,66('*'))
                  WRITE(6,83)
                  DO 84 J=1,M
85                  FORMAT(D13.5,2X)
                    WRITE(6,85)T(J)
86                  FORMAT((14X,3(D20.10,1X)))
                    WRITE(6,86)(X(L1),L1=(J-1)*N+1,J*N)
84                CONTINUE
                ENDIF
              ENDIF
9997        CONTINUE
C.          End of Segment Bvpsol.Core
9998      CONTINUE
        GOTO 16
        ENDIF
C.      EndWhile
C
C       End of main iteration loop
C       ==========================
C
C       ----------------------------------------------------------
C       5 Exits
C       ----------------------------------------------------------
C       5.1 Solution exit
        IF(INFO.EQ.0)THEN
          ITER = ITER+1
C:        Vec X = Vec X + Vec DXQ (for 1,NM)
          DO 87 L1=1,NM
            X(L1)=X(L1)+DXQ(L1)
87        CONTINUE
C.        End SetVec.&Vec
          INFO = ITER
          IF(KPRINT.LT.0)THEN
            GOTO 9999
          ENDIF
          IF(KPRINT.GT.0)THEN
C           Print levels, damping factor of last iteration step
88          FORMAT('0',1X,66('*'))
            WRITE(6,88)
89          FORMAT('0',4X,'It',4X,'Ny',7X,'Levelf',10X,'Levelx',8X,
     *      'Rel.Fc.')
            WRITE(6,89)
          ENDIF
90        FORMAT('0',4X,I2,4X,I2,5X,D10.3,2X,4X,D10.3,2X,4X,F5.3)
          WRITE(6,90)ITER,NY,SUMF,SUMX,FC
91        FORMAT('0',1X,66('*'))
          WRITE(6,91)
92        FORMAT('1')
          WRITE(6,92)
          IF(IRANK.LT.NE)THEN
            INFO = -1
          ELSE
C           Print solution info
93          FORMAT('0','Solution ','of',1X,'boundary ','value ',
     *      'problem',' obtained',/,'0','BVPSOL',' required',I3,1X,
     *      'Iteration ','steps ','with',I4,1X,'trajectory',
     *      ' evaluations',//)
            WRITE(6,93)ITER,KOUNT
            CALL BLPRCV(CONV,EPH)
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       5.2 Fail exit messages
C       ----------------------------------------------------------
C       5.2.1 Rank-deficiency : best least squares solution of bvp
C             obtained
        IF(INFO.EQ.-1.AND.KPRINT.GE.0)THEN
94        FORMAT('0','Iteration ','terminates ','at ',
     *    'stationary ','point',/)
          WRITE(6,94)
          CALL BLPRCV(CONVA,EPH)
          IF(ITER.NE.0)THEN
            SKAP = ZERO
            IF(FCA.EQ.ONE.AND.FC.EQ.ONE.AND.IRANKA.EQ.IRANK) SKAP
     *      = DSQRT(SUMXA/FCNUMK)
            IF(SKAP.GT.ZERO)THEN
95            FORMAT('0','Incompatibility ','factor ','kappa',D10.3
     *        ,2X,/)
              WRITE(6,95)SKAP
            ENDIF
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       5.2.2 Termination after more than itmax iterations
        IF(INFO.EQ.-2.AND.KPRINT.GE.0)THEN
96        FORMAT('0','Iteration ','terminates ','after ','itmax ',
     *    '=',I3,2X,'iteration ','steps')
          WRITE(6,96)ITMAX
        ENDIF
C       ----------------------------------------------------------
C       5.2.3 Singular trajectory
        IF(INFO.EQ.-3.AND.KPRINT.GE.0)THEN
97        FORMAT('0','Singular ','trajectory ','by ','difference ',
     *    'approximation ','of ','the ','jacobian ','matrix',/)
          WRITE(6,97)
          J1 =-KFLAG
98        FORMAT('0','BVPSOL ','terminates',/,'Subinterval',I3,1X,
     *    'possibly ','insert ','new ','node',D20.11,2X,/)
          WRITE(6,98)J1,TFAIL
        ENDIF
C       ----------------------------------------------------------
C       5.2.4 Convergence fail of Gauss - Newton method
        IF(INFO.EQ.-4.AND.KPRINT.GE.0)THEN
99        FORMAT('0','Gauss ','Newton ','method ','fails ','to ',
     *    'converge',/)
          WRITE(6,99)
        ENDIF
C       ----------------------------------------------------------
C       5.2.5 Inconsistent initial data
        IF(INFO.EQ.-5.AND.KPRINT.GE.0)THEN
100       FORMAT('0','Error: ','initial ','data ','and ',
     *    'boundary ','conditions ','are ','inconsistent',/)
          WRITE(6,100)
        ENDIF
C       ----------------------------------------------------------
C       5.2.6 Convergence fail of iterative refinement sweeps
        IF(INFO.EQ.-6)THEN
          IF(KPRINT.GE.0)THEN
101         FORMAT('0','Termination ','since ','iterative ',
     *      'refinement ','fails ','to ','converge',/,2X,'Insert ',
     *      'new ','nodes',/)
            WRITE(6,101)
          ENDIF
          JN = JN-1
          IF(JN.GT.0)THEN
102         FORMAT('0',8X,'in ','subinterval',2X,I3,/)
            WRITE(6,102)JN
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       5.2.7 Insufficient error tolerance for integrator
        IF(INFO.EQ.-7.AND.KPRINT.GE.0)THEN
          TOLH = EPS/SIGDEL
          RELDIF = DSQRT(TOLH/SIGDEL)
103       FORMAT('0','Suggested ','integrator ','accuracy',D10.1
     *    ,2X,/,'0','Suggested ','relative ','deviation ',
     *    'parameter',D10.1,2X,/)
          WRITE(6,103)TOLH,RELDIF
104       FORMAT('0','Reduce ','relative ','error ','tolerance ',
     *    'for ','integrator ','to',D10.1,2X,/,2X,'or ','insert ',
     *    'new ','nodes',/)
          WRITE(6,104)TOLH
          S = REDH/TOL
          DO 105 J=1,M1
            IF(RF(J).GT.S)THEN
106           FORMAT(2X,'in ','subinterval',I3,/)
              WRITE(6,106)J
            ENDIF
105       CONTINUE
107       FORMAT('0','Reliable ','relative ','accuracy ',
     *    'greater ','than',1X,D6.1,2X,/)
          WRITE(6,107)1.0D-2
        ENDIF
C       ----------------------------------------------------------
C       5.2.8 ill - conditioned condensed linear system
        IF(INFO.EQ.-8.AND.KPRINT.GE.0)THEN
108       FORMAT('0','Gaussian ','block ','elimination ','fails',/,
     *    2X,'by ','ill ','- ','conditioned ','condensed ',
     *    'linear ','system',/)
          WRITE(6,108)
          IF(IRANK.EQ.NE)THEN
109         FORMAT('0','Relative ','accuracy ','of ','DX1',D10.3
     *      ,2X)
            WRITE(6,109)EPX1H
          ENDIF
110       FORMAT('0','Possibly ','turn ','to ','code ','BVPSOG ',
     *    'instead ','of ','BVPSOL',/)
          WRITE(6,110)
        ENDIF
C       ----------------------------------------------------------
C       5.3 Common exit
        IF(KPRINT.GE.0)THEN
C
          J1 = 1
          SMALIN = ONE/SMALL
          IF(IRANKB.NE.0) CALL BLPRCD(COND1,SENS1,SMALIN,J1,IRANKB)
          IF(IRANKB.NE.IRANK)THEN
            J1 = IRANKB+1
            CALL BLPRCD(COND2,SENS2,SMALIN,J1,IRANK)
          ENDIF
111       FORMAT('0','Multiple ','shooting ','condition',D10.3,2X,
     *    /,'1')
          WRITE(6,111)SIGDLH
          IF(INFO.GT.0)THEN
112         FORMAT('0','Solution ','data',/)
            WRITE(6,112)
          ENDIF
          IF(INFO.LT.0)THEN
113         FORMAT('0','Final ','data',/)
            WRITE(6,113)
          ENDIF
          DO 114 J=1,M
115         FORMAT(D13.5,2X)
            WRITE(6,115)T(J)
116         FORMAT((14X,3(D20.10,1X)))
            WRITE(6,116)(X(L1),L1=(J-1)*N+1,J*N)
114       CONTINUE
        ENDIF
C       End of exits
C       End of subroutine BVPSOL
9999  CONTINUE
C.    End of Segment Bvpsol.Body
      RETURN
      END
      SUBROUTINE BLFCNI(IVPSOL,FCN,BC,N,M,NM,NM1,ITER,KPRINT,
     *HSTART,FCMIN,T,X,X1,XM,T1,XU,HH,R,TOL,FC,FCOMPT,REDUCT,KFLAG,
     *KOUNT,INFO)
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL FCN,IVPSOL,BC
      INTEGER N,M,NM,NM1,ITER,KPRINT
      DOUBLE PRECISION HSTART,FCMIN
      INTEGER KOUNT,KFLAG,INFO
      LOGICAL REDUCT
      DOUBLE PRECISION TOL,FC
      LOGICAL FCOMPT
      DOUBLE PRECISION T(M),X(NM)
      DOUBLE PRECISION XU(NM1),HH(NM1),R(N),X1(N),XM(N),T1(N)
C:    End Parameter
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      INTEGER J,J1,KB,KB2
      DOUBLE PRECISION HMAX,HSAVE,TJ,TJ1,H
      INTEGER L1
C:    Begin
C:    Begin of Segment FcnInt.Body
C       Computation of the trajectories (solution of M1 initial
C         value problems)
        KOUNT = KOUNT+1
        HSAVE = HSTART
        DO 117 J=1,M-1
          J1 = J+1
          TJ = T(J)
          TJ1 = T(J1)
          H = HSAVE
          HMAX = DABS(TJ1-TJ)
          KFLAG = 0
          KB =(J-1)*N
C:        Begin SetVec.Vec
          DO 118 L1=1,N
            T1(L1)=X(L1+KB)
118       CONTINUE
C.        End SetVec.Vec
          CALL IVPSOL(N,FCN,TJ,T1,TJ1,TOL,HMAX,H,KFLAG)
          HSAVE = H
          IF(H.EQ.ZERO)THEN
C           singular trajectory
            IF(ITER.EQ.0)THEN
              INFO = -3
              GOTO 9993
            ENDIF
            IF(KPRINT.GE.0)THEN
119           FORMAT('0','Singular ','trajectory, ','relaxation ',
     *        'factor ','or ','pseudo-rank ','reduced',/)
              WRITE(6,119)
            ENDIF
            FC = FC*HALF
            IF(FC.LT.FCMIN)THEN
              REDUCT = .TRUE.
              GOTO 9993
            ENDIF
            FCOMPT = .FALSE.
            GOTO 9993
          ENDIF
          FCOMPT = .TRUE.
C         continuity conditions
C:        Begin SetVec.Vec
          DO 120 L1=1,N
            XU(L1+KB)=T1(L1)
120       CONTINUE
C.        End SetVec.Vec
          KB2 = KB+N
C:        Begin SetVec.Vec-Vec
          DO 121 L1=1,N
            HH(L1+KB)=T1(L1)-X(L1+KB2)
121       CONTINUE
C.        End SetVec.Vec-Vec
117     CONTINUE
C       two-point boundary conditions
C:      Begin SetVec.Vec
        DO 122 L1=1,N
          XM(L1)=X(L1+NM1)
122     CONTINUE
C.      End SetVec.Vec
C:      Begin SetVec.Vec
        DO 123 L1=1,N
          X1(L1)=X(L1)
123     CONTINUE
C.      End SetVec.Vec
        CALL BC(X1,XM,R)
9993  CONTINUE
C.    End of Segment FcnInt.Body
      RETURN
      END
      SUBROUTINE BLSOLI(N,M,M1,NM,NM1,LEVEL,NE,NB,IRANK,IRANKB,
     *IREPET,NYMAX,KPRINT,EPS,REDH,TOLMIN,TOL,RELDIF,EPH,EPX1H,
     *SIGDEL,SIGDLH,E,EH,HH,DHH,R,A,B,BG,G,QE,U,QU,DE,DU,T1,T2,US,
     *DX1,D,DDX,DXQ,XW,DR,RF,IROW,ICOL,ICOLB,PIVOT,NY,INFO)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N,M,M1,NM,NM1,LEVEL,NE,NB,IRANK,IRANKB,IREPET,NYMAX,
     *KPRINT
      DOUBLE PRECISION EPS,REDH,TOLMIN
      DOUBLE PRECISION TOL,RELDIF,EPH,EPX1H,SIGDEL,SIGDLH
      INTEGER INFO,NY
      INTEGER IROW(N),ICOL(N),ICOLB(N),PIVOT(N)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION A(N,N),B(N,N),BG(N,N),E(N,N),QE(N,N)
      DOUBLE PRECISION DDX(NM),DXQ(NM),XW(NM),HH(NM1),DHH(NM1),D(N),
     *DE(N),R(N),DR(N),U(N),DU(N),QU(N),T1(N),T2(N),DX1(N),RF(M),
     *US(N)
      DOUBLE PRECISION EH(N,N)
C:    End Parameter
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=1.0D1)
      INTEGER I,IC,I0,J,JA,JIN,JN,J1,K,K0
      DOUBLE PRECISION CORR,EPDX1,EPX1,S,TH,TOLH
      LOGICAL NOTFND
      INTEGER L1,L2
      DOUBLE PRECISION S1
C:    Begin
C:    Begin of Segment SolvIn.Body
        IF(IREPET.GE.0)THEN
C         --------------------------------------------------------
C         1 computation of condensed right - hand side U(NE)
          IF(IRANK.GT.0) CALL BLRHS1(N,NE,M1,NM1,1,HH,R,B,G,U,DE,
     *    T1,BG,IROW)
C         --------------------------------------------------------
C         2 saving of right - hand side U
          IF(IRANK.GE.NE)THEN
C:          Begin SetVec.Vec
            DO 124 L1=1,NE
              US(L1)=U(L1)
124         CONTINUE
C.          End SetVec.Vec
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       3 ( best ) constrained least squares solution of linear(NE,
C         NE) -system
C:      Vec DX1 = Scalar (for 1,N)
        S1 = ZERO
        DO 125 L1=1,N
          DX1(L1)=S1
125     CONTINUE
C.      End SetVec.S
        IF(IRANK.GT.0)THEN
          CALL SOLCON(E,N,N,IRANKB,NE,NE,DX1,U,IRANK,D,PIVOT,
     *    IREPET,QE,T1)
        ENDIF
C       ----------------------------------------------------------
C       4 iterative refinement of DX1
        EPH = EPS
        IF(IRANK.GE.NE.AND.NE.NE.0)THEN
C:        Begin SetVec.MatxVec
          DO 126 L1=1,NE
            S1=0.0
            DO 127 L2=1,NE
              S1=S1+EH(L1,L2)*DX1(L2)
127         CONTINUE
            DU(L1)=S1
126       CONTINUE
C.        End SetVec.MatxVec
C:        Begin SetVec.Vec-Vec
          DO 128 L1=1,NE
            DU(L1)=US(L1)-DU(L1)
128       CONTINUE
C.        End SetVec.Vec-Vec
C         Solution of residual equation
          CALL SOLCON(E,N,N,IRANKB,NE,NE,T2,DU,IRANK,D,PIVOT,
     *    IREPET,QE,T1)
C:        EPDX1 = Max.Norm of Vec T2 (for 1,NE)
          EPDX1 = 0.0
          DO 129 L1=1,NE
            S1 = DABS(T2(L1))
            IF(S1.GT.EPDX1) EPDX1=S1
129       CONTINUE
C.        End MakeMaxNorm.Vec
C:        Vec DX1 = Vec DX1 + Vec T2 (for 1,NE)
          DO 130 L1=1,NE
            DX1(L1)=DX1(L1)+T2(L1)
130       CONTINUE
C.        End SetVec.&Vec
C:        EPX1 = Max.Norm of Vec DX1 (for 1,NE)
          EPX1 = 0.0
          DO 131 L1=1,NE
            S1 = DABS(DX1(L1))
            IF(S1.GT.EPX1) EPX1=S1
131       CONTINUE
C.        End MakeMaxNorm.Vec
          EPX1H = EPDX1/EPX1
          EPH = TEN*EPDX1
          IF(EPX1H.GT.HALF)THEN
            INFO = -8
            GOTO 9992
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       5 Descaling and back - permutation of solution DX1
C:      Permutation ICOL of Vec DXQ = Vec DX1 (for 1,N)
        DO 132 L1=1,N
          L2 = ICOL(L1)
          DXQ(L2) = DX1(L1)
132     CONTINUE
C.      End SetVecByPermVec.Vec
C:      Vec DXQ = Vec DXQ * Vec XW (for 1,N)
        DO 133 L1=1,N
          DXQ(L1)=DXQ(L1)*XW(L1)
133     CONTINUE
C.      End SetVec.VecxVec
C       ----------------------------------------------------------
C       6 Recursive computation of DXQ(N,2),  ... ,  DXQ(N,M)
        CALL BLRCRS(N,M,M1,NM,NM1,1,HH,G,DXQ,T1,T2)
C       ----------------------------------------------------------
C       1 Iterative refinement sweeps NY = 1 ,  ... ,  NYMAX
        NY = 0
        SIGDEL = TEN*TEN
        SIGDLH = ZERO
        IF(EPH.LT.EPS) EPH = EPS
        IF(NYMAX.NE.0.AND.IRANK.GE.NE.AND.NE.NE.0)THEN
          IF(KPRINT.GT.0)THEN
134         FORMAT('0','Iterative ','refinement',/)
            WRITE(6,134)
          ENDIF
C         --------------------------------------------------------
C         1.1 Computation of required continuity residuals DHH(N,
C             M1)
          JN = 1
          JIN = M
C         --------------------------------------------------------
C         1.2 Computation of boundary residual DR(N)
C:        DO (Until)
135       CONTINUE
C:          Begin SetVec.MatxVec
            DO 136 L1=1,N
              S1=0.0
              DO 137 L2=1,N
                S1=S1+A(L1,L2)*DXQ(L2)
137           CONTINUE
              DR(L1)=S1
136         CONTINUE
C.          End SetVec.MatxVec
C:          Begin SetVec.MatxVec
            DO 138 L1=1,N
              S1=0.0
              DO 139 L2=1,N
                S1=S1+B(L1,L2)*DXQ(L2+NM1)
139           CONTINUE
              T1(L1)=S1
138         CONTINUE
C.          End SetVec.MatxVec
C:          Vec DR = Formula (for 1,N)
            DO 140 L1=1,N
              DR(L1)=R(L1)+DR(L1)+T1(L1)
140         CONTINUE
C.          End SetVec.Comp
C           ------------------------------------------------------
C           1.3 Computation of condensed residual DU(NE)
            IF(IRANK.GT.0) CALL BLRHS1(N,NE,M1,NM1,JIN,DHH,DR,B,G,
     *      DU,DE,T1,BG,IROW)
C           ------------------------------------------------------
C           1.4 Computation of correction DDX(N)
C:          Vec DX1 = Scalar (for 1,N)
            S1 = ZERO
            DO 141 L1=1,N
              DX1(L1)=S1
141         CONTINUE
C.          End SetVec.S
            IF(IRANK.GT.0) CALL SOLCON(E,N,N,IRANKB,NE,NE,DX1,DU,
     *      IRANK,D,PIVOT,IREPET,QE,T1)
C           ------------------------------------------------------
C           2 Descaling of DDX(N),  refinement of DXQ(N)
C:          CORR = Max.Norm of Vec DX1 (for 1,N)
            CORR = 0.0
            DO 142 L1=1,N
              S1 = DABS(DX1(L1))
              IF(S1.GT.CORR) CORR=S1
142         CONTINUE
C.          End MakeMaxNorm.Vec
C:          Permutation ICOL of Vec T1 = Vec DX1 (for 1,N)
            DO 143 L1=1,N
              L2 = ICOL(L1)
              T1(L2) = DX1(L1)
143         CONTINUE
C.          End SetVecByPermVec.Vec
C:          Vec DDX = Vec T1 * Vec XW (for 1,N)
            DO 144 L1=1,N
              DDX(L1)=T1(L1)*XW(L1)
144         CONTINUE
C.          End SetVec.VecxVec
C:          Vec DXQ = Vec DXQ + Vec DDX (for 1,N)
            DO 145 L1=1,N
              DXQ(L1)=DXQ(L1)+DDX(L1)
145         CONTINUE
C.          End SetVec.&Vec
            IF(CORR.GE.EPH)THEN
              EPH = CORR
              INFO = -8
              GOTO 9992
            ENDIF
            RF(1)=CORR
C           ------------------------------------------------------
C           3 Recursive computation of DDX(N+1),  ... ,  DDX(NM)
            CALL BLRCRS(N,M,M1,NM,NM1,JIN,DHH,G,DDX,T1,T2)
C           ------------------------------------------------------
C           3.1 Refinement of DXQ(N+1),  ... ,  DXQ(NM)
            DO 146 J=2,M
              I0 =(J-1)*N
C:            Vec DXQ = Vec DXQ + Vec DDX (for I0+1,I0+N)
              DO 147 L1=I0+1,I0+N
                DXQ(L1)=DXQ(L1)+DDX(L1)
147           CONTINUE
C.            End SetVec.&Vec
C:            CORR = Max of Formula Elements (for I0+1,I0+N)
              CORR =-7.23D75
              DO 148 L1=I0+1,I0+N
                S1=DABS(DDX(L1)/XW(L1))
                IF(S1.GT.CORR)CORR=S1
148           CONTINUE
C.            End MakeMax.Comp
              RF(J)=CORR
146         CONTINUE
C           ------------------------------------------------------
C           3.2 Determination of sweep index JN
            JA = JN
            J = 1
            NOTFND = .TRUE.
C:          While (expression)
149         IF(J.LE.M.AND.NOTFND)THEN
              IF(RF(J).GT.EPH)THEN
                NOTFND = .FALSE.
              ELSE
                JN = J
                J = J+1
              ENDIF
            GOTO 149
            ENDIF
C.          EndWhile
            NY = NY+1
            IF(KPRINT.GT.0)THEN
150           FORMAT('0','Sweep',1X,I3,1X,'starts ','at',1X,I3)
              WRITE(6,150)NY,JA
151           FORMAT((1X,5(D12.3,1X)))
              WRITE(6,151)(RF(L1),L1=1,M)
            ENDIF
            IF(JN.LE.JA)THEN
              INFO = -6
              GOTO 9992
            ENDIF
            IF(JN.NE.M) JIN = JN
C           ------------------------------------------------------
C           3.3 Determination and adaptation of parameters TOL AND
C               RELDIF
            IF(LEVEL.NE.0.AND.NY.LE.1)THEN
              DO 152 J=1,M1
                S = RF(J+1)/RF(J)
                IF(SIGDLH.LT.S) SIGDLH = S
                RF(J)=S
152           CONTINUE
              IF(KPRINT.GT.0)THEN
153             FORMAT('0','Norms ','of ','wronskians')
                WRITE(6,153)
154             FORMAT((1X,5(D12.3,1X)))
                WRITE(6,154)(RF(L1),L1=1,M1)
              ENDIF
              SIGDEL = DMAX1(SIGDLH,SIGDEL)
              TH = TOL*SIGDEL
              IF(TH.GT.REDH)THEN
                INFO = -7
                GOTO 9992
              ENDIF
              IF(TH.GT.EPH) EPH = TH
              TOLH = EPS/SIGDEL
              IF(TOLH.LT.TOLMIN) TOLH = TOLMIN
CWei;         TOL = TOLH
CWei;         RELDIF = DSQRT(TOL/SIGDEL)
              IF(KPRINT.GE.0)THEN
155             FORMAT('0','Suggested ','integrator ','accuracy',D10.1
     *          ,2X,/,'0','Suggested ','relative ','deviation ',
     *          'parameter',D10.1,2X,//,'0','Adapted ','in ',
     *          'the ','next ','iteration ','step',/)
                WRITE(6,155)TOLH,RELDIF
              ENDIF
            ENDIF
            IF(JN.NE.M)THEN
              DO 156 J=JN,M1
                J1 = J+1
                DO 157 I=1,N
                  K0 =(J-1)*N
                  S = HH(I+K0)
                  DO 158 K=1,N
                    S = S+G(I,K,J)*DXQ(K+K0)
158               CONTINUE
                  DHH(I+K0)=S-DXQ(I+K0+N)
157             CONTINUE
156           CONTINUE
            ENDIF
          IF(.NOT.(JN.EQ.M)) GOTO  135
C.        UNTIL ( expression - negated above)
        ENDIF
C       End of iterative refinement sweeps
C       ----------------------------------------------------------
C       4 Projection of separated linear boundary conditions at T(
C         M)
        IF(NB.NE.0)THEN
          DO 159 K=1,NB
            IC = ICOLB(K)
            DXQ(IC+NM1)=ZERO
159       CONTINUE
        ENDIF
9992  CONTINUE
C.    End of Segment SolvIn.Body
      RETURN
      END
      SUBROUTINE BLPRCD(COND,SENS,SMALIN,J,IRANK)
      IMPLICIT DOUBLEPRECISION(S)
      DOUBLE PRECISION COND,SENS,SMALIN
      INTEGER J,IRANK
C:    End Parameter
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION SENSP
C:    Begin
      IF(SENS.LT.ONE)THEN
        SENSP = SENS*SMALIN
160     FORMAT('0','Subcondition (',I2,',',I2,') ',D10.3,2X,/,'0',
     *  'Sensitivity  (',I2,',',I2,') ',D10.3,2X,/)
        WRITE(6,160)J,IRANK,COND,J,IRANK,SENSP
      ELSE
161     FORMAT('0','Subcondition ','(',I2,',',I2,') ',D10.3,2X,/,
     *  '0','Sensitivity ','(',I2,',',I2,') ',D10.3,2X,' *',D7.0
     *  ,2X,/)
        WRITE(6,161)J,IRANK,COND,J,IRANK,SENS,SMALIN
      ENDIF
      END
      SUBROUTINE BLPRCV(CONV,EPH)
      IMPLICIT DOUBLEPRECISION(S)
      DOUBLE PRECISION CONV,EPH
C:    End Parameter
C:    Begin
162   FORMAT('0','Achieved ','relative ','accuracy',D10.3,2X)
      WRITE(6,162)CONV
      IF(EPH.GT.CONV) CONV = EPH
163   FORMAT('0','Reliable ','relative ','accuracy',D10.3,2X,/)
      WRITE(6,163)CONV
      END
      SUBROUTINE BLSCLE(N,M,NM,NM1,X,XU,XW,XTHR)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N
      INTEGER M
      INTEGER NM
      INTEGER NM1
      DOUBLE PRECISION X(NM)
      DOUBLE PRECISION XW(NM)
      DOUBLE PRECISION XU(NM1)
      DOUBLE PRECISION XTHR
C:    End Parameter
C     PROVIDES SCALING XW(NM)OF VARIABLES X(NM)
C:    EPMACH = relative machine precision
      DOUBLE PRECISION EPMACH
      PARAMETER (EPMACH=2.23D-16)
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION RED
      PARAMETER (RED=1.0D-2)
      INTEGER I,J,J0,J1
      DOUBLE PRECISION XMAX
      INTEGER L1
C:    Begin
C:    Vec XW = Formula (for 1,N)
      DO 164 L1=1,N
        XW(L1)=DABS(X(L1))
164   CONTINUE
C.    End SetVec.Comp
C     ------------------------------------------------------------
C     1 Arithmetic mean for XW(2*N)... XW(M*N)
      DO 165 J=1,M-1
        J0 =(J-1)*N
        J1 = J0+N
        DO 166 I=1,N
          XW(I+J1)=(DABS(X(I+J1))+DABS(XU(I+J0)))*HALF
166     CONTINUE
165   CONTINUE
C     ------------------------------------------------------------
C     2 Threshold
      DO 167 I=1,N
        XMAX = ZERO
        DO 168 J=0,NM1,N
          IF(XMAX.LT.XW(I+J)) XMAX = XW(I+J)
168     CONTINUE
        XMAX = XMAX*RED
        IF(XMAX.LT.XTHR) XMAX = XTHR
        DO 169 J=0,NM1,N
          IF(XW(I+J).LT.XMAX) XW(I+J)=XMAX
169     CONTINUE
167   CONTINUE
      RETURN
C     End of subroutine BLSCLE
      END
      SUBROUTINE BLLVLS(N,M,NM,NM1,XW,DXQ,HH,R,DE,CONV,SUMX,SUMF,
     *KPRINT)
      IMPLICIT DOUBLEPRECISION(S)
C
      INTEGER N,M,NM,NM1,KPRINT
      DOUBLE PRECISION XW(NM),DXQ(NM),HH(NM1),R(N),DE(N)
      DOUBLE PRECISION CONV,SUMX,SUMF
C:    End Parameter
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,J,J0,J1
      DOUBLE PRECISION S
      INTEGER L1
C:    Begin
C     ------------------------------------------------------------
C     1 Evaluation of scaled natural level function SUMX and
C       scaled maximum error norm CONV
      CONV = ZERO
      SUMX = ZERO
      DO 170 J=1,NM
        S = DABS(DXQ(J)/XW(J))
        IF(CONV.LT.S) CONV = S
        SUMX = SUMX+S*S
170   CONTINUE
C     ------------------------------------------------------------
C     2 Evaluation of (scaled) standard level function sumfs (only
C       if needed for print)
C:    SUMF = Sum of Formula Elements (for 1,N)
      SUMF = 0.0D0
      DO 171 L1=1,N
        SUMF=SUMF+((R(L1)*DE(L1)/SMALL)**2)
171   CONTINUE
C.    End MakeSum.Comp
      DO 172 J=1,M-1
        J0 =(J-1)*N
        J1 = J0+N
        DO 173 I=1,N
          SUMF = SUMF+(HH(I+J0)/XW(I+J1))**2
173     CONTINUE
172   CONTINUE
C     End of subroutine BLLVLS
      RETURN
      END
      SUBROUTINE BLRHS1(N,NE,M1,NM1,JIN,HH,R,B,G,U,DE,V,BG,IROW)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N,NE,M1,NM1,JIN
      DOUBLE PRECISION HH(NM1),R(N)
      DOUBLE PRECISION B(N,N)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION U(N),DE(N),V(N)
      DOUBLE PRECISION BG(N,N)
      INTEGER IROW(N)
C:    End Parameter
C     Computation of condensed right-hand side U(NE)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,IR,J,JJ,J1,K,K0,L,M2
      DOUBLE PRECISION S,TH
C:    Begin
      DO 174 I=1,NE
        IR = IROW(I)
        U(I)=DE(IR)*R(IR)
174   CONTINUE
      IF(JIN.GT.M1)THEN
        RETURN
      ENDIF
      DO 175 I=1,NE
        IR = IROW(I)
        S = U(I)
        K0 = NM1-N
        DO 176 K=1,N
          TH = DE(IR)*B(IR,K)
          BG(I,K)=TH
          S = S+TH*HH(K+K0)
176     CONTINUE
        U(I)=S
175   CONTINUE
      IF(M1.EQ.1.OR.JIN.EQ.M1) RETURN
      M2 = M1-1
      DO 177 JJ=JIN,M2
        J = M2+JIN-JJ
        J1 = J+1
        DO 178 I=1,NE
          DO 179 K=1,N
            S = ZERO
            DO 180 L=1,N
              S = S+BG(I,L)*G(L,K,J1)
180         CONTINUE
            V(K)=S
179       CONTINUE
          S = U(I)
          K0 =(J-1)*N
          DO 181 K=1,N
            S = S+V(K)*HH(K+K0)
            BG(I,K)=V(K)
181       CONTINUE
          U(I)=S
178     CONTINUE
177   CONTINUE
C     End of subroutine BLRHS1
      RETURN
      END
      SUBROUTINE BLRCRS(N,M,M1,NM,NM1,JIN,HH,G,DX,U,V)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N,M,M1,NM,NM1,JIN
      DOUBLE PRECISION HH(NM1)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION DX(NM),U(N),V(N)
C:    End Parameter
C     Recursive solution of m1 linear(N,N)-systems
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,J,J0,J1,K
      INTEGER L1
      DOUBLE PRECISION S
C:    Begin
C:    Begin SetVec.Vec
      DO 182 L1=1,N
        U(L1)=DX(L1)
182   CONTINUE
C.    End SetVec.Vec
      DO 183 J=1,M1
        J0 =(J-1)*N
        J1 = J0+N
        DO 184 I=1,N
          IF(J.GE.JIN)THEN
            S = HH(I+J0)
          ELSE
            S = ZERO
          ENDIF
          DO 185 K=1,N
            S = S+G(I,K,J)*U(K)
185       CONTINUE
          V(I)=S
          DX(I+J1)=S
184     CONTINUE
C:      Begin SetVec.Vec
        DO 186 L1=1,N
          U(L1)=V(L1)
186     CONTINUE
C.      End SetVec.Vec
183   CONTINUE
C     End of subroutine BLRCRS
      RETURN
      END
      SUBROUTINE BLPRJC(N,NE,IRANK,DEL,U,D,V,QE,PIVOT)
      IMPLICIT DOUBLEPRECISION(S)
C
      INTEGER IRANK,N,NE
      INTEGER PIVOT(N)
      DOUBLE PRECISION DEL
      DOUBLE PRECISION QE(N,N)
      DOUBLE PRECISION U(N),D(N),V(N)
C:    End Parameter
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER L1,L2
      INTEGER I,IRK1
      DOUBLE PRECISION S,SH
C:    Begin
      DO 187 I=1,NE
        V(I)=U(PIVOT(I))
187   CONTINUE
      IRK1 = IRANK+1
      DEL = ZERO
      DO 188 I=IRK1,NE
C:      SH = Col I of QE * Vec V (for 1,I-1)
        SH = 0.0
        DO 189 L1=1,I-1
          SH = SH+QE(L1,I)*V(L1)
189     CONTINUE
C.      End MakeSProd.ColxVec
        S =(V(I)-SH)/D(I)
        DEL = DEL-S*S
        V(I)=S
188   CONTINUE
      DO 190 I=IRK1,NE
        K = NE+IRK1-I
        S = V(K)
        IF(K.NE.NE)THEN
C:        SH = Row K of QE * Vec V (for K+1,NE)
          SH = 0.0
          DO 191 L1=K+1,NE
            SH=SH+QE(K,L1)*V(L1)
191       CONTINUE
C.        End MakeSProd.RowxVec
          S = S-SH
        ENDIF
        S = S/D(K)
        V(K)=S
190   CONTINUE
      DO 192 I=1,IRANK
C:      S = Row I of QE * Vec V (for IRK1,NE)
        S = 0.0
        DO 193 L1=IRK1,NE
          S=S+QE(I,L1)*V(L1)
193     CONTINUE
C.      End MakeSProd.RowxVec
        V(I)=-S
192   CONTINUE
C:    Permutation PIVOT of Vec U = Vec V (for 1,NE)
      DO 194 L1=1,NE
        L2 = PIVOT(L1)
        U(L2) = V(L1)
194   CONTINUE
C.    End SetVecByPermVec.Vec
C     End of subroutine BLPRJC
      RETURN
      END
      SUBROUTINE BLDERA(BC,N,M,NM,XW,X1,XM,R,RH,A,B,RELDIF)
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL BC
      INTEGER N,M,NM
      DOUBLE PRECISION XW(NM),X1(N),XM(N),R(N),RH(N)
      DOUBLE PRECISION A(N,N),B(N,N)
      DOUBLE PRECISION RELDIF
C:    End Parameter
C     Difference approx. of boundary derivative matrices A(N,N)and
C       B(N,N)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,K,NM1
      DOUBLE PRECISION S,XH
C:    Begin
      NM1 = N*(M-1)
      DO 195 K=1,N
        XH = X1(K)
        S = RELDIF*XW(K)
        IF(XH.LT.ZERO) S =-S
        X1(K)=XH+S
        CALL BC(X1,XM,RH)
        X1(K)=XH
        S = ONE/S
        DO 196 I=1,N
          A(I,K)=(RH(I)-R(I))*S
196     CONTINUE
        XH = XM(K)
        S = RELDIF*XW(K+NM1)
        IF(XH.LT.ZERO) S =-S
        XM(K)=XH+S
        CALL BC(X1,XM,RH)
        XM(K)=XH
        S = ONE/S
        DO 197 I=1,N
          B(I,K)=(RH(I)-R(I))*S
197     CONTINUE
195   CONTINUE
C     END SUBROUTINE BLDERA
      RETURN
      END
      SUBROUTINE BLDERG(FCN,N,NE,M,M1,NM,NM1,T,X,XU,XW,XJ,TJ,G,
     *ICOL,IVPSOL,HSTART,TOL,RELDIF,KFLAG)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N
      INTEGER NE
      INTEGER M
      INTEGER M1
      INTEGER NM
      INTEGER NM1
      DOUBLE PRECISION T(M)
      DOUBLE PRECISION X(NM)
      DOUBLE PRECISION XU(NM1)
      DOUBLE PRECISION XW(NM)
      DOUBLE PRECISION XJ(N)
      DOUBLE PRECISION TJ
      DOUBLE PRECISION G(N,N,M1)
      INTEGER ICOL(N)
      EXTERNAL IVPSOL
      DOUBLE PRECISION HSTART
      DOUBLE PRECISION TOL
      DOUBLE PRECISION RELDIF
      INTEGER KFLAG
C:    End Parameter
C     Difference approximation of Wronskian Matrices G(1),.., G(M1)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,IK,J,J0,J1,K
      DOUBLE PRECISION HMAX,H,HSAVE,S,TJ1,TJA,TH
      EXTERNAL FCN
C:    Begin
      HSAVE = HSTART
      DO 198 J=1,M-1
        J0 =(J-1)*N
        J1 = J+1
        TJA = T(J)
        TJ1 = T(J1)
        HMAX = DABS(TJ1-TJA)
        DO 199 IK=1,N
          I = ICOL(IK)
          H = HSAVE
          IF(J.NE.1.OR.IK.LE.NE)THEN
            TJ = TJA
            KFLAG = 0
            DO 200 K=1,N
              XJ(K)=X(K+J0)
200         CONTINUE
            TH = XJ(I)
            S = RELDIF*XW(I+J0)
            IF(TH.LT.ZERO) S =-S
            XJ(I)=TH+S
            S = ONE/S
            CALL IVPSOL(N,FCN,TJ,XJ,TJ1,TOL,HMAX,H,KFLAG)
            IF(H.EQ.ZERO)THEN
              KFLAG =-J
              RETURN
            ENDIF
            DO 201 K=1,N
              G(K,I,J)=S*(XJ(K)-XU(K+J0))
201         CONTINUE
          ENDIF
199     CONTINUE
        HSAVE = H
198   CONTINUE
      KFLAG = 0
C     END OF SUBROUTINE BLDERG
      RETURN
      END
      SUBROUTINE BLRK1G(N,M,M1,NM,NM1,XW,DX,HH,HHA,DXJ,G,FCA)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N,M,M1,NM,NM1
      DOUBLE PRECISION XW(NM),DX(NM),HH(NM1),HHA(NM1),DXJ(N)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION FCA
C:    End Parameter
C     RANK-1 UPDATES OF WRONSKIAN MATRICES G(1),..., G(M1)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,J,J0,K
      DOUBLE PRECISION DNM,FCH,S,T
C:    Begin
      FCH = FCA-ONE
      DO 202 J=1,M1
        J0 =(J-1)*N
        DNM = ZERO
        DO 203 I=1,N
          T = DX(I+J0)/XW(I+J0)
          DXJ(I)=T/XW(I+J0)
          DNM = DNM+T*T
203     CONTINUE
        DNM = DNM*FCA
        IF(DNM.NE.ZERO)THEN
          DO 204 K=1,N
            T = DXJ(K)/DNM
            DO 205 I=1,N
              S = G(I,K,J)
              IF(S.NE.ZERO) G(I,K,J)=S+T*(HH(I+J0)+FCH*HHA(I+J0))
205         CONTINUE
204       CONTINUE
        ENDIF
202   CONTINUE
C     END OF SUBROUTINE BLRK1G
      RETURN
      END
C
C*    Group  Initial value problem solver (Code DIFEX1)   
C
      SUBROUTINE DIFEX1 (N,FCN,T,Y,TEND,TOL,HMAX,H,KFLAG)
C
C* Begin Prologue DFEX1
C
C  ---------------------------------------------------------------------
C
C* Title
C
C  Explicit Extrapolation Integrator
C  for Non-Stiff Systems of Ordinary First-Order Differential Equations
C
C* Written by        P. Deuflhard, U. Nowak, U. Poehle
C* Purpose           Solution of systems of initial value problems
C* Method            Implicit mid-point rule discretization with
C                    h**2-extrapolation
C* Category          i1a1a. - System of nonstiff first order
C                             differential equations
C* Keywords          extrapolation, ODE, explicit mid-point rule,
C                    nonstiff
C* Version           1.0 , February 1988
C* Latest Change     January 1991
C* Library           CodeLib
C* Code              Fortran 77
C                    Double Precision
C* Environment       Standard version for FORTRAN77 environments on
C                    PCs, workstations, and hosts
C* Copyright     (c) Konrad-Zuse-Zentrum fuer Informationstechnik
C                    Berlin (ZIB)
C                    Heilbronner Str. 10, D-1000 Berlin 31
C                    phone:   0049+30/89604-0
C                    telefax: 0049+30/89604-125
C* Contact           ZIB
C                    Numerical Software Development
C                    Uwe Poehle
C                    phone:   0049+30/89604-184
C                    e-mail:  poehle@sc.zib-berlin.de
C
C  ---------------------------------------------------------------------
C
C* Licence
C  -------
C
C  You may use or modify this code for your own non-commercial
C  purposes for an unlimited time. 
C  In any case you should not deliver this code without a special 
C  permission of ZIB.
C  In case you intend to use the code commercially, we oblige you
C  to sign an according licence agreement with ZIB.
C
C
C* Warranty
C  --------
C 
C  This code has been tested up to a certain level. Defects and
C  weaknesses, which may be included in the code, do not establish
C  any warranties by ZIB. ZIB does not take over any liabilities
C  which may follow from aquisition or application of this code.
C
C
C* Software status 
C  ---------------
C
C  This code is under care of ZIB and belongs to ZIB software
C  class I.
C
C
C  ---------------------------------------------------------------------
C
C* References:
C
C /1/ W. B. Gragg:
C     On Extrapolation Algorithms for Ordinary
C     Initial Value Problems
C     SIAM J. Numer. Anal. 2, 384-404 (1965)
C
C /2/ R. Bulirsch, J. Stoer:
C     Numerical Treatment of Ordinary Differential Equations
C     by Extrapolation Methods
C     Num. Math. 8, 1-13 (1966)
C
C /3/ P. Deuflhard:
C     Order and Stepsize Control in Extrapolation Methods
C     Numer. Math. 41, 399-422 (1983)
C
C
C* External Subroutine: (to be Supplied by the User)
C
C    FCN           EXT  Subroutine FCN (N,T,Y,DY)
C                       Right-Hand Side of First-Order
C                       Differential Equations
C                       N      Number of First-Order ODE'S
C                       T      Actual Position
C                       Y(N)   Values at T
C                       DY(N)  Derivatives at T
C
C
C* Parameters: (* Marks Inout Parameters)
C
C    N         I   IN   Number of First-Order ODE'S
C  * T         D   IN   Starting Point of Integration
C                       (T .LE. TEND)
C                  OUT  Achieved Final Point of Integration
C  * Y         D   IN   Array of Initial Values Y(1),...,Y(N)
C                  OUT  Array of Final Values
C    TEND      D   IN   Prescribed Final Point of Integration
C    TOL       D   IN   Prescribed Relative Precision (.GT.0)
C    HMAX      D   IN   Maximum Permitted Stepsize
C  * H         D   IN   Initial Stepsize Guess
C                  OUT  Stepsize Proposal for Next Integration Step
C                       (H .EQ. 0. ,if DIFEX1 Fails to Proceed)
C  * KFLAG     I   IN   Print Parameter
C                        0   no Output
C                        1   Integration Monitor
C                        2   Intermediate Solution Points  T,Y(I),I=1,N
C                        3   Integration Monitor and Solution Points
C                  OUT  Error Flag
C                       .GE. 0  Successful Integration
C                               (KFLAG not Altered Internally)
C                       -1   TEND .LT. T
C                       -2   More Than NSTMAX Basic Integration Steps
C                            per Interval Have Been Performed
C                       -3   More Than JRMAX Stepsize Reductions
C                            Occurred per Basic Integration Step
C                       -4   Stepsize Proposal for Next Basic
C                            Integration Step too Small
C
C    COMMON /STAT/ NFCN, NSTEP, NACCPT, NREJCT, NDEC, NSOL
C                       Internally Initialized, for Statistical
C                       Purposes
C    NFCN               Number of FCN-Evaluatios
C    NSTEP              Number of Integration Steps
C    NACCPT             Number of Steps Accepted
C    NREJCT             Number of Steps Rejected
C    NDEC               Number of Decompositions
C    NSOL               Number of Substitutions
C
C* Type Declaration
C
      INTEGER I, J, JK, JL, JM, JMACT, JOPT, JRED, JRMAX, J1, K, KFIN,
     2KFLAG, KM, KMACT, KOPT, K1, L, LOUT, M, MAXODE, MDT, M1, N,
     3NACCPT, NDEC, NFCN, NJ, NREJCT, NSOL, NSTEP, NSTMAX
C
      DOUBLE PRECISION ALPHA, AWK, B1, C, D, DBLE, DMAX1, DMIN1, DT,
     2DUMMY, DY, DYM, DZ, D1ERRN, EPMACH, EPSAFE, ERR, FC, FCK, FCM,
     3FCO, FMIN, FNJ, FN1, FN, H, HALF, HJ,  HJ2, HMAX, HMAXU, HN,
     4HREST, HRTRN, OMJ, OMJO, ONE, PCT101, PCT90, QUART, RED, RMAX, RO,
     5SAFE, SMALL, T, TEND, TN, TOL, TOLH, TOLMIN, U, V, W, Y, YA, YK,
     6YM, YMAX, YWGT, ZERO
C
      LOGICAL QFIRST, QKONV, QLAST, QPRMON, QPRSOL, QRED
C
      CHARACTER CHGDAT*20, PRODCT*8
C
C
C* Constants Problem Oriented: (to be Supplied by the User)
C
C    MAXODE    I   K    Maximal Number of First-Order ODE'S
C
      PARAMETER ( MAXODE = 51            )
C
C* Constants Machine Oriented: (to be Verified by the User)
C  (Adapted to Siemens 7.865, IBM 370-Compatible)
C
C    EPMACH    D   K    Relative Machine Precision
C    LOUT      I   K    Output is Written on Logical Unit LOUT
C    SMALL     D   K    Square-Root of Smallest Positive Machine Number
C
      PARAMETER ( EPMACH = 2.22D-16      ,
     2            EPSAFE = EPMACH*10.0D0 ,
     3            LOUT   = 6             ,
     4            SMALL  = 7.35D-40      )
C
C* Other Constants:
C
C    HALF      D   K    1/2
C    ONE       D   K    1
C    PCT101    D   K    101 Percent
C    PCT90     D   K    90 Percent
C    QUART     D   K    1/4
C    ZERO      D   K    0
C
      PARAMETER ( HALF   = 0.5  D0       ,
     2            ONE    = 1.0  D0       ,
     3            PCT101 = 1.01 D0       ,
     4            PCT90  = 0.9  D0       ,
     5            QUART  = 0.25 D0       ,
     6            ZERO   = 0.0  D0       )
C
C* Control Parameters: (to be Supplied by the User)
C  Standard Values Fixed Below
C
C    NSTMAX    I   K    Maximum Permitted Number of Integration Steps
C                       per Interval  =  10000
C    JRMAX     I   K    Maximum Permitted Number of Stepsize Reductions
C    KM        I   K    Prescribed Maximum Column Number
C    JM        I   K    Associated Maximum Row Number
C                       (JM = KM + 1)
C    MDT       I   K    Associated Dimension of DT
C    D1SEQ         EXT  Subroutine D1SEQ(JM,NJ)
C                       Generate Stepsize Sequence with Respect to /1/
C                       JM     Maximum Row Number
C                       NJ     Array(JM) of Stepsize Sequence
C    D1SCAL        EXT  Subroutine D1SCAL (MODE, Y, N, YOLD, YWGT,
C                                          YMAX, THREL, THABS)
C                       Scaling for DIFEX1
C                       MODE   ='INITIAL '    Initial Scaling
C                              ='INTERNAL'    Scaling during Discret.
C                              ='ACCEPTED'    Rescaling if Step Accepted
C                              Else           Error
C                       Y      Array of Values Y(1),...,Y(N)
C                       N      Length of Vectors Y, YOLD, YWGT, and YMAX
C                       YOLD   Array of Old Values
C                       YWGT   Array of Scaled Values
C                       YMAX   Array of Maximum Values
C                       THREL  Relative Threshold Value
C                       THABS  Absolute Threshold Value
C    D1ERRN        EXT  Double Precision Function D1ERRN(Y, N, YWGT)
C                       Scaled Root Mean Square Error
C                       Y      Array of Values Y(1),...,Y(N)
C                       N      Length of Vectors Y and YWGT
C                       YWGT   Array of Scaled Values
C
      PARAMETER ( NSTMAX = 10000         ,
     2            JRMAX  = 10            ,
     3            KM     = 8             ,
     4            JM     = KM + 1        ,
     5            MDT    = MAXODE*JM     )
C
C* Internal Parameters: (Modification not Recommended)
C
C
      PARAMETER ( FMIN   = 1.0  D-3      ,
     2            RMAX   = 0.75 D0       ,
     3            RO     = QUART         ,
     4            SAFE   = 0.7  D0       )
C
C
C* Local Variables: (Workspace)
C
C
C
C    QFIRST    L   V    First Integration Step
C    QKONV     L   V    Convergence Detected
C    QLAST     L   V    Last Integration Step
C    QPRMON    L   V    Print Integration Monitor
C    QPRSOL    L   V    Print Intermediate Solution Points
C
C* Dimensions:
C
      DIMENSION ALPHA(JM,JM), AWK(JM), D(JM,JM), DT(MAXODE,JM),
     2DY(MAXODE), DYM(MAXODE), DZ(MAXODE), FCK(KM), NJ(JM),
     3Y(MAXODE), YK(MAXODE), YM(MAXODE), YMAX(MAXODE), YWGT(MAXODE)
C
      COMMON /STAT/ NFCN, NSTEP, NACCPT, NREJCT, NDEC, NSOL
C
C*******  Revision 1 *******  Latest Change:
      DATA      CHGDAT      /'February 25, 1988   '/
      DATA      PRODCT      /'DIFEX1'/
C***************************
C
C
      DATA  DT/MDT*0.D0/
C
C---1. Initial Preparations
      QPRMON = (KFLAG .EQ. 1 .OR. KFLAG .EQ. 3)
      QPRSOL = (KFLAG .GE. 2)
      IF (TEND .LT. T) THEN
C        Error 1
         IF (QPRMON) WRITE (LOUT, 10001) PRODCT, T, TEND
         KFLAG = -1
         GOTO 9
C        Exit to Return
      ENDIF
      DO 1001 I = 1, N
 1001    YMAX(I) = ZERO
C     ENDDO
      HREST = TEND - T
      H = DMIN1 (H, HREST)
      HMAXU = HMAX
      IF (HMAX .GT. EPSAFE) THEN
         FCM = DMAX1(H/HMAX, FMIN)
      ELSE
         FCM = FMIN
      ENDIF
      KMACT = KM
      JMACT = JM
      CALL D1SEQ (JM, NJ)
      FN = DBLE (N)
      FN1 = DBLE (NJ(1))
      TOLH = RO*TOL
      TOLMIN = EPSAFE*FN
      IF (TOL .LT. TOLMIN) THEN
         WRITE (LOUT, 10002) PRODCT, TOL, TOLMIN
         TOL = TOLMIN
      ENDIF
C
C---  Compute Amount of Work per Row of Extrapolation Tableau
      AWK(1) = FN1 + ONE
      DO 101 J=2,JM
         J1 = J - 1
         FNJ = DBLE (NJ(J))
         V = AWK(J1) + FNJ
         AWK(J) = V
         DO 1011 K=1,J1
 1011       D(J,K) = (FNJ / DBLE (NJ(K)))*(FNJ / DBLE (NJ(K)))
C        ENDDO
         IF (J .NE. 2) THEN
            W = V - AWK(1) + ONE
            DO 1012 K1=2,J1
               K = K1 - 1
               U = (AWK(K1) - V) / (W*DBLE(K + K1))
               U = TOLH**U
 1012          ALPHA(J1,K) = U
C           ENDDO
         ENDIF
 101     CONTINUE
C     ENDDO
C
C---1.2 Determination of Maximum Column Number in Extrapolation
C---    Tableau (Information Theoretic Concept, Ref./3/)
      KOPT = 1
      JOPT = 2
 121  CONTINUE
C     DO WHILE (JOPT .LT. KMACT .AND.
C               AWK(JOPT+1)*PCT101 .LE. AWK(JOPT)*ALPHA(JOPT,KOPT))
         IF (JOPT .GE. KMACT .OR.
     2      AWK(JOPT+1)*PCT101 .GT. AWK(JOPT)*ALPHA(JOPT,KOPT)) GOTO 122
C                                                         Exit 121
         KOPT = JOPT
         JOPT = JOPT + 1
         GOTO  121
C     ENDDO
 122  KMACT = KOPT + 1
      JMACT = JOPT
      IF (QPRMON) WRITE (LOUT, 11221)
     2   PRODCT, CHGDAT, TOL, KMACT, NJ
C
      IF (QPRSOL) WRITE (LOUT, 11222)
      NSTEP = 0
      QFIRST = .TRUE.
      QLAST = .FALSE.
CWEI; NFCN = 0
      KFIN = 0
      OMJO = ZERO
      CALL D1SCAL ('INITIAL ', Y, N, DUMMY, YWGT, YMAX, TOL, ONE)
C
C---2. Basic Integration Step
 2    CONTINUE
C     DO WHILE (T .NE. TEND)
         IF (QPRMON) WRITE (LOUT, 12001) NSTEP,NFCN,T,KFIN,KOPT
         IF (QPRSOL) WRITE (LOUT, 12002) NSTEP,NFCN,T,H,(Y(I),I=1,N)
         JRED = 0
C
C---     Explicit Euler Starting Step
         CALL FCN (N, T, Y, DZ)
         NFCN = NFCN + 1
C
C---3.   Basic Discretization Step
 3       CONTINUE
C        DO WHILE (JRED .LE. JRMAX .AND. .NOT. QKONV)
            IF (QLAST) THEN
               TN = TEND
            ELSE
               TN = T + H
            ENDIF
            IF (TN .EQ. T) THEN
C              Error 4
               IF (QPRMON) WRITE (LOUT, 13001) PRODCT
               KFLAG = -4
               GOTO  9
C              Exit to Return
            ENDIF
C
C---3.1     Internal Discretization
            DO 31 J=1,JMACT
               M = NJ(J)
               M1 = M - 1
               KFIN = J - 1
               FNJ = DBLE (M)
               HJ = H / FNJ
               HJ2 = HJ + HJ
               DO 3101 I=1,N
                  YK(I) = Y(I)
 3101             YM(I) = Y(I) + HJ*DZ(I)
C              ENDDO
C
C---3.1.3      Explicit Mid-Point Rule
               DO 313 K=1,M1
                  CALL FCN (N, T + HJ*DBLE (K), YM, DY)
                  NFCN = NFCN + 1
                  DO 3135 I=1,N
                     U = YK(I) + HJ2*DY(I)
                     YK(I) = YM(I)
 3135                YM(I) = U
C                 ENDDO
 313              CONTINUE
C              ENDDO
C
C---3.1.4      Smoothing Final Step
               CALL FCN (N, TN, YM, DY)
               NFCN = NFCN + 1
               DO 3141 I = 1,N
 3141             YM(I) = (YM(I) + YK(I) + HJ*DY(I))*HALF
C              ENDDO
C
C---3.1.5      Extrapolation
               ERR = ZERO
               DO 315 I=1,N
                  C = YM(I)
                  V = DT(I,1)
                  DT(I,1) = C
                  IF (J .NE. 1) THEN
                     YA = C
                     DO 3151 K=2,J
                        JK = J - K + 1
                        B1 = D(J,JK)
                        W = C - V
                        U = W / (B1 - ONE)
                        C = B1*U
                        V = DT(I,K)
                        DT(I,K) = U
 3151                   YA = U + YA
C                    ENDDO
                     YM(I) = YA
                     DYM(I) = U
                  ENDIF
 315              CONTINUE
C              ENDDO
               IF (J .NE. 1) THEN
C
C---3.1.6         Convergence Monitor
                  CALL D1SCAL ('INTERNAL',YM,N,Y,YWGT,YMAX,TOL,ONE)
                  ERR = D1ERRN (DYM, N, YWGT)
                  QKONV = ERR .LE. TOL
                  ERR = ERR / TOLH
C
C---              Order Control
                  K = J - 1
                  FC = ERR**(ONE / DBLE(K + J))
                  FCK(K) = FC
C
C---              Order Window
                  IF (J .GE. KOPT .OR. QFIRST .OR. QLAST) THEN
                     IF (QKONV) GOTO 25
C                                Exit 3 for Next Basic Integration Step
C
C---                 Check for Possible Stepsize Reduction
                     RED = ONE / FC
                     QRED = .FALSE.
                     IF (K .EQ. KMACT .OR. K .EQ. JOPT) THEN
                        RED = RED*SAFE
                        QRED = .TRUE.
                     ELSE
                        IF (K .EQ. KOPT) THEN
                           RED = RED*ALPHA(JOPT,KOPT)
                           IF (RED .LT. ONE) THEN
                              RED = ONE / FC
                              QRED = .TRUE.
                           ENDIF
                        ELSE
                           IF (KOPT .EQ. KMACT) THEN
                              RED = RED*ALPHA(KMACT,K)
                              IF (RED .LT. ONE) THEN
                                 RED = RED * SAFE
                                 QRED = .TRUE.
                              ENDIF
                           ELSE
                              RED = RED*ALPHA(JOPT,K)
                              IF (RED .LT. ONE) THEN
                                 RED = ALPHA(KOPT,K) / FC
                                 QRED = .TRUE.
                              ENDIF
                           ENDIF
                        ENDIF
                     ENDIF
                     IF (QRED) GOTO 32
C                              Exit 3.1 to Stepsize Reduction
                  ENDIF
               ENDIF
 31            CONTINUE
C           ENDDO
C
C---3.2     Prepare Stepsize Reduction
 32         CONTINUE
C
C---3.5     Stepsize Reduction
            RED = DMIN1 (RED, RMAX)
            H = H*RED
            IF (NSTEP .GT. 0) QLAST = .FALSE.
            JRED = JRED + 1
            IF (QPRMON) WRITE (LOUT, 13501) JRED,RED,
     2         KFIN,KOPT,KMACT
            IF (JRED .GT. JRMAX) THEN
C              Error 3
               IF (QPRMON) WRITE (LOUT, 13502) JRMAX
               KFLAG = -3
               GOTO  9
C              Exit to Return
            ENDIF
            GOTO  3
C        ENDDO
C
C        ************************************************
C---2.5  Preparations for Next Basic Integration Step
 25      NSTEP = NSTEP + 1
         QFIRST = .FALSE.
         IF (NSTEP .GT. NSTMAX) THEN
C           Error 2
C           Emergency Exit, If too Many Steps Taken
            IF (QPRMON) WRITE (LOUT, 12501) PRODCT, NSTMAX
            KFLAG = -2
            GOTO  9
C           Exit to Return
         ENDIF
C
C---     Restoring
         DO 251 I=1, N
 251        Y(I) = YM(I)
C        ENDDO
         T = TN
         IF (T .EQ. TEND) GOTO 9
C                         Exit to Return
         CALL D1SCAL ('ACCEPTED', Y, N, DUMMY, YWGT, YMAX, TOL, ONE)
C
C---2.7  Order and Stepsize Selection
C
C---2.7.1 Stepsize Restrictions
         HMAX = DMIN1(HMAXU,H/FMIN)
         FCM = H / HMAX
C
C---2.7.2 Optimal Order Determination
         KOPT = 1
         JOPT = 2
         FCO = DMAX1 (FCK(1), FCM)
         OMJO = FCO*AWK(2)
         IF (KFIN .GE. 2) THEN
            DO 272 L=2,KFIN
               JL = L + 1
               FC = DMAX1 (FCK(L), FCM)
               OMJ = FC*AWK(JL)
               IF (OMJ*PCT101 .LE. OMJO .AND. L .LT. KMACT) THEN
                  KOPT = L
                  JOPT = JL
                  OMJO = OMJ
                  FCO = FC
               ENDIF
 272           CONTINUE
C           ENDDO
         ENDIF
         HREST = TEND - T
         HN = H / FCO
C
C---2.7.3 Possible Increase of Order
         IF (HN .LT. HREST) THEN
            IF ((JRED .EQ. 0 .OR. NSTEP .EQ. 0) .AND.
     2           KOPT .GE. KFIN .AND. KOPT .NE. KMACT) THEN
               FC = DMAX1 (FCO/ALPHA(JOPT,KOPT), FCM)
               JL = JOPT + 1
               IF (AWK(JL)*FC*PCT101 .LE. OMJO .AND.
     2               JOPT .LT. KMACT) THEN
                  FCO = FC
                  HN = H / FCO
                  KOPT = JOPT
                  JOPT = JOPT + 1
               ENDIF
            ENDIF
         ENDIF
C
C---2.7.4 Stepsize Selection
         H = HN
         HRTRN = H
         IF (H .GT. HREST*PCT90) THEN
            H = HREST
            QLAST = .TRUE.
         ENDIF
         GO TO  2
C     ENDDO
C
C---9. Exit
 9    HMAX = HMAXU
      IF (KFLAG .LT. 0) THEN
C        Fail Exit
         H = ZERO
      ELSE
C        Solution Exit
         H = HRTRN
         IF (QPRMON) WRITE (LOUT, 12001) NSTEP,NFCN,T,KFIN,KOPT
         IF (QPRSOL) WRITE (LOUT, 12002) NSTEP,NFCN,T,H,(Y(I),I=1,N)
      ENDIF
      RETURN
C
C
10001 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,   ' Direction of integration is reverse to convention.')
10002 FORMAT(//,' ',A8,'  - WARNING -'
     2      ,   ' Desired tolerance ', D10.3, ' too small.', /,
     3      22X,' Tolerance set to  ', D10.3, '.')
11221 FORMAT(1H0,A8,' - ',A20,/,
     2       1H0,' rel.prec. TOL ',D10.3,' max.col.',I3,
     3       ' Sequence ',(1H ,13I4))
11222 FORMAT(//,5X,4HStep,3X,7HF-Calls,8X,1HT,25X,1HH,5X,7HY1(T)..,//)
12001 FORMAT(1H ,2I9,D20.11,I9,I6)
12002 FORMAT(1H ,2I9,D20.11,D12.5,4D20.11,/,(1H ,50X,4D20.11))
12501 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,18H More than NSTMAX=,I3,18H integration steps,//)
13001 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,40H Stepsize reduction failed to succeed  ,//)
13501 FORMAT(1H ,I3,27H Stepsize reduction factor ,D10.3,
     2      ' KFIN',I3,' KOPT',I3,' KMAX',I3)
13502 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,17H More then JRMAX=,I3,29H stepsize reductions per step,/)
C
C
C End DIFEX1
C
      END
      SUBROUTINE D1SEQ(M,NJ)
      INTEGER I, M, NJ
      DIMENSION NJ(M)
C
C  Set Stepsize Sequence for DIFEX1
C
      NJ(1) = 2
      DO 10 I=2,M
        NJ(I) = NJ(I-1) + 2
 10     CONTINUE
C     ENDDO
      RETURN
      END
      SUBROUTINE D1SCAL (MODE, Y, N, YOLD, YWGT, YMAX, THREL, THABS)
C
C     Scaling for DIFEX1
C
C       (for Real Life Applications to be Altered
C        by the Skillful User)
C
C
C* Parameters:
C
C    MODE      C*8 IN   ='INITIAL '    Initial Scaling
C                       ='INTERNAL'    Scaling during Discretization
C                       ='ACCEPTED'    Rescaling if Step Accepted
C                       Else           Error
C    Y         D   IN   Array of Values Y(1),...,Y(N)
C    N         I   IN   Length of Vectors Y, YOLD, YWGT, and YMAX
C    YOLD      D   IN   Array of Old Values
C    YWGT      D   OUT  Array of Scaled Values New
C    YMAX      D   IN   Array of Maximum Values Old
C                  OUT  Array of Maximum Values New
C    THREL     D   IN   Relative Threshold Value
C    THABS     D   IN   Absolute Threshold Value
C
C* Local Variables:
C
C    YUSER     D   V    User Defined Array of Maximum Values
C
C* Type Declaration
C
      INTEGER I, LOUT, MAXODE, N
C
      DOUBLE PRECISION DABS, DMAX1, EPMACH, ONE, THABS, THREL, U, Y,
     2YMAX, YOLD, YUSER, YWGT, ZERO
C
      CHARACTER MODE*8
C
C* Constants:
C
C    EPMACH    D   K    Relative Machine Precision
C    LOUT      I   K    Output is Written on Logical Unit LOUT
C    MAXODE    I   K    Maximal Number of First-Order ODE's
C    ONE       D   K    1.0
C    ZERO      D   K    0.0
C
      PARAMETER ( EPMACH = 2.22D-16      ,
     2            LOUT   = 6             ,
     3            MAXODE = 51            ,
     4            ONE    = 1.0  D0       ,
     5            ZERO   = 0.0  D0       )
C
      DIMENSION Y(N), YOLD(N), YWGT(N), YMAX(N), YUSER(MAXODE)
      SAVE YUSER
      IF (MODE .EQ.          'INITIAL '         ) THEN
C                             --------
         DO 100 I=1,N
            YUSER(I) = DABS (YMAX(I))
            U = DABS (Y(I))
            IF (U .LT. EPMACH) U = ONE
            YMAX(I) = DMAX1 (U, YUSER(I), THABS)
 100        YWGT(I) = YMAX(I)
C        ENDDO
      ELSE IF (MODE .EQ.     'INTERNAL'         ) THEN
C                             --------
         DO 200 I=1,N
 200        YWGT(I) = DMAX1 (YMAX(I)*THREL, DABS(Y(I)),
     2                       DABS(YOLD(I)), YUSER(I), THABS)
C        ENDDO
      ELSE IF (MODE .EQ.     'ACCEPTED'         ) THEN
C                             --------
         DO 300 I=1,N
 300        YMAX(I) = DMAX1 (YMAX(I), DABS(Y(I)))
C        ENDDO
      ELSE
         WRITE (LOUT, '(//,A,/)')
     2      ' D1SCAL    - ERROR -   Illegal Mode'
      ENDIF
      RETURN
      END
      DOUBLE PRECISION FUNCTION D1ERRN(Y, N, YWGT)
C* Title:
C
C  Scaled Root Mean Square Error
C
C
C* Parameters:
C
C    Y         D   IN   Array of Values Y(1),...,Y(N)
C    N         I   IN   Length of Vectors Y and YWGT
C    YWGT      D   IN   Array of Scaled Values
C
C* Type Declaration
C
      INTEGER I, N
C
      DOUBLE PRECISION DBLE, DSQRT, SUM, Y, YWGT, ZERO
C
C* Constants:
C
C    ZERO      D   K    0
C
      PARAMETER ( ZERO   = 0.0  D0       )
C
      DIMENSION Y(N), YWGT(N)
C
      SUM = ZERO
      DO 100 I=1,N
 100     SUM = SUM + (Y(I) / YWGT(I)) * (Y(I) / YWGT(I))
C     ENDDO
      D1ERRN = DSQRT(SUM / DBLE(N))
      RETURN
      END
C
C*    Group  Linear Solver subroutines (Code DECCON/SOLCON)
C
      SUBROUTINE DECCON (A,NROW,NCOL,MCON,M,N,IRANK,COND,D,
     1                                            PIVOT,KRED,AH,V)
C     ------------------------------------------------------------
C
C*  Title
C
C*    Deccon - Constrained Least Squares QR-Decomposition
C
C*  Written by        P. Deuflhard
C*  Purpose           Solution of least squares problems, optionally
C                     with equality constraints.
C*  Method            Constrained Least Squares QR-Decomposition
C                     (see references below)
C*  Category          D9b1. -  Singular, overdetermined or
C                              underdetermined systems of linear 
C                              equations, generalized inverses. 
C                              Constrained Least Squares solution
C*  Keywords          Linear Least Square Problems, constrained, 
C                     QR-decomposition, pseudo inverse.
C*  Version           0.9
C*  Revision          April 1984
C*  Latest Change     January 1991
C*  Library           CodeLib
C*  Code              Fortran 77, Double Precision
C*  Environment       Standard Fortran 77 environment on PC's,
C                     workstations and hosts.
C*  Copyright     (c) Konrad Zuse Zentrum fuer
C                     Informationstechnik Berlin
C                     Heilbronner Str. 10, D-1000 Berlin 31
C                     phone 0049+30+89604-0, 
C                     telefax 0049+30+89604-125
C*  Contact           Lutz Weimann 
C                     ZIB, Numerical Software Development 
C                     phone: 0049+30+89604-185 ;
C                     e-mail:
C                     RFC822 notation: weimann@sc.zib-berlin.de
C                     X.400: C=de;A=dbp;P=zib-berlin;OU=sc;S=Weimann
C
C*    References:
C     ===========
C
C       /1/ P.Deuflhard, V.Apostolescu:
C           An underrelaxed Gauss-Newton method for equality
C           constrained nonlinear least squares problems.
C           Lecture Notes Control Inform. Sci. vol. 7, p.
C           22-32 (1978)
C       /2/ P.Deuflhard, W.Sautter:
C           On rank-deficient pseudoinverses.
C           J. Lin. Alg. Appl. vol. 29, p. 91-111 (1980)
C    
C*    Related Programs:     SOLCON
C
C  ---------------------------------------------------------------
C
C* Licence
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C* Warranty 
C    This code has been tested up to a certain level. Defects and
C    weaknesses, which may be included in the code, do not establish
C    any warranties by ZIB. ZIB does not take over any liabilities
C    which may follow from aquisition or application of this code.
C
C* Software status 
C    This code is under partial care of ZIB and belongs to ZIB software
C    class 2.
C
C     ------------------------------------------------------------
C
C*    Summary:
C     ========
C     Constrained QR-decomposition of (M,N)-system  with
C     computation of pseudoinverse in case of rank-defeciency .
C     First MCON rows belong to equality constraints.
C
C     ------------------------------------------------------------
C
C     INPUT PARAMETERS (* MARKS INOUT PARAMETERS)
C     -----------------------------------------------
C
C
C      * A(NROW,NCOL)  INPUT MATRIX
C                      A(M,N) CONTAINS ACTUAL INPUT
C        NROW          DECLARED NUMBER OF ROWS OF A AND AH
C        NCOL          DECLARED NUMBER OF COLUMNS OF A AND AH
C     (*)MCON          NUMBER OF EQUALITY CONSTRAINTS (MCON<=N)
C                      INTERNALLY REDUCED IF EQUALITY CONSTRAINTS
C                      ARE LINEARLY DEPENDENT
C        M             TREATED NUMBER OF ROWS OF MATRIX A
C        N             TREATED NUMBER OF COLUMNS OF MATRIX A
C     (*)IRANK         PSEUDO-RANK OF MATRIX A
C      * COND          PERMITTED UPPER BOUND OF DABS(D(1)/D(IRANKC))
C                      AND OF DABS(D(IRANKC+1))/D(IRANK))
C                      (SUB-CONDITION NUMBERS OF A)
C        KRED          >=0    HOUSEHOLDER TRIANGULARIZATION
C                             (BUILD UP OF PSEUDO-INVERSE,IF IRANK<N )
C                      < 0    REDUCTION OF PSEUDO-RANK OF MATRIX A
C                             SKIPPING HOUSEHOLDER TRIANGULARIZATION
C                             BUILD-UP OF NEW PSEUDO-INVERSE
C        V(N)          REAL WORK ARRAY
C
C     OUTPUT PARAMETERS
C     -----------------
C
C        A(M,N)        OUTPUT MATRIX UPDATING PRODUCT OF HOUSEHOLDER
C                      TRANSFORMATIONS AND UPPER TRIANGULAR MATRIX
C        MCON          PSEUDO-RANK OF CONSTRAINED PART OF MATRIX A
C        IRANK         PSEUDO-RANK OF TOTAL MATRIX A
C        D(IRANK)      DIAGONAL ELEMENTS OF UPPER TRIANGULAR MATRIX
C        PIVOT(N)      INDEX VECTOR STORING PERMUTATION OF COLUMNS
C                      DUE TO PIVOTING
C        COND          SUB-CONDITION NUMBER OF A
C                      (IN CASE OF RANK REDUCTION: SUB-CONDITION NUMBER
C                      WHICH LED TO RANK REDUCTION)
C        AH(N,N)       UPDATING MATRIX FOR PART OF PSEUDO INVERSE
C
C----------------------------------------------------------------------
C
      INTEGER  IRANK, KRED, MCON, M, N, NROW, NCOL, PIVOT(N)
      INTEGER  I, II, IRK1, I1, J, JD, JJ, K, K1, MH, ISUB
      DOUBLE PRECISION    A(NROW,NCOL), AH(NCOL,NCOL), D(N), V(N)
      DOUBLE PRECISION    COND, ONE , DD, DABS, DSQRT
      DOUBLE PRECISION    H, HMAX, S, T, SMALL, ZERO, EPMACH
C     COMMON /MACHIN/ EPMACH, SMALL
C
      PARAMETER( ZERO=0.D0, ONE=1.D0 )
C
C  RELATIVE MACHINE PRECISION
C  ADAPTED TO IBM 370/168 (UNIVERSITY OF HEIDELBERG)
      PARAMETER( EPMACH = 2.2D-16 )
C
      SMALL = DSQRT(EPMACH*1.D1)
C
      IF(IRANK.GT.N) IRANK=N
      IF(IRANK.GT.M) IRANK=M
C
C---1.0 SPECIAL CASE M=1 AND N=1
C
      IF(M.EQ.1 .AND. N.EQ.1) THEN
         PIVOT(1)=1
         D(1)=A(1,1)
         COND=1.D0
         RETURN
      ENDIF
C
C---1.1 INITIALIZE PIVOT-ARRAY
      IF  (KRED.GE.0)  THEN
         DO 1100 J=1,N
1100        PIVOT(J) = J
C        ENDDO
C
C
C---2. CONSTRAINED HOUSEHOLDER TRIANGULARIZATION
C
         JD = 1
         ISUB = 1
         MH = MCON
         IF (MH.EQ.0) MH=M
         K1 = 1
2000     K = K1
         IF (K.NE.N)  THEN
            K1 = K+1
2100        IF (JD.NE.0)  THEN
               DO  2110 J=K,N
                  S = ZERO
                  DO 2111 I=K,MH
2111                 S = S+A(I,J)*A(I,J)
C                 ENDDO
2110              D(J) = S
C              ENDDO
            ENDIF
C
C---2.1     COLUMN PIVOTING
            H = D(K)
            JJ = K
            DO   2120 J=K1,N
               IF (D(J).GT.H)  THEN
                  H = D(J)
                  JJ = J
               ENDIF
2120        CONTINUE
C           ENDDO
            IF (JD.EQ.1)  HMAX = H * SMALL
            JD = 0
            IF (H.LT.HMAX)  THEN
               JD = 1
               GOTO 2100
            ENDIF
            IF (JJ.NE.K)  THEN
C
C---2.2        COLUMN INTERCHANGE
               I = PIVOT(K)
               PIVOT(K) = PIVOT(JJ)
               PIVOT(JJ) = I
               D(JJ) = D(K)
               DO  2210 I=1,M
                  T = A(I,K)
                  A(I,K) = A(I,JJ)
2210              A(I,JJ) = T
C              ENDDO
            ENDIF
         ENDIF
C
         H = ZERO
         DO  2220 I=K,MH
2220        H = H+A(I,K)*A(I,K)
C        ENDDO
         T = DSQRT(H)
C
C---2.3.0  A PRIORI TEST ON PSEUDO-RANK
C
         IF (ISUB.GT.0) DD = T/COND
         ISUB = 0
         IF (T.LE.DD) THEN
C
C---2.3.1 RANK REDUCTION
C
            IF (K.LE.MCON) THEN
C              CONSTRAINTS ARE LINEARLY DEPENDENT
               MCON = K-1
               K1 = K
               MH = M
               JD = 1
               ISUB = 1
               GOTO 2000
            ENDIF
C
            IRANK = K - 1
            IF (IRANK.EQ.0)  THEN
               GOTO 4000
            ELSE
               GOTO 3000
            ENDIF
         ENDIF
C
         S = A(K,K)
         IF (S.GT.ZERO) T = -T
         D(K) = T
         A(K,K) = S-T
         IF (K.EQ.N)  GOTO 4000
C
         T = ONE/(H-S*T)
         DO  2300 J=K1,N
            S = ZERO
            DO  2310 I=K,MH
2310           S = S+A(I,K)*A(I,J)
C           ENDDO
            S = S*T
            DO  2320 I=K,M
2320           A(I,J) = A(I,J)-A(I,K)*S
C           ENDDO
2300        D(J) = D(J)-A(K,J)*A(K,J)
C        ENDDO
C
         IF (K.EQ.IRANK) GOTO 3000
         IF (K.EQ.MCON) THEN
            MH = M
            JD = 1
            ISUB = 1
         ENDIF
         GOTO 2000
      ENDIF
C
C---3. RANK-DEFICIENT PSEUDO-INVERSE
C
3000  IRK1 = IRANK+1
      DO  3300 J=IRK1,N
         DO  3100 II=1,IRANK
            I = IRK1-II
            S = A(I,J)
            IF (II.NE.1)  THEN
               DO  3110 JJ=I1,IRANK
3110              S = S-A(I,JJ)*V(JJ)
C              ENDDO
            ENDIF
            I1 = I
            V(I) = S/D(I)
3100        AH(I,J) = V(I)
C        ENDDO
         DO  3200 I=IRK1,J
            S = ZERO
            I1 = I-1
            DO  3210 JJ=1,I1
3210           S = S+AH(JJ,I)*V(JJ)
C           ENDDO
            IF (I.NE.J)  THEN
               V(I) = -S/D(I)
               AH(I,J) = -V(I)
            ENDIF
3200     CONTINUE
C        ENDDO
3300     D(J) = DSQRT(S+ONE)
C     ENDDO
C
C---4.  EXIT
C
4000  IF (K.EQ.IRANK) T=D(IRANK)
      IF (T.NE.0.D0) COND=DABS(D(1)/T)
      RETURN
C
C     **********  LAST CARD OF DECCON  **********
C
      END
C
      SUBROUTINE SOLCON (A,NROW,NCOL,MCON,M,N,X,B,IRANK,D,
     @                   PIVOT,KRED,AH,V)
C
C
C     BEST CONSTRAINED LINEAR LEAST SQUARES SOLUTION OF (M,N)-SYSTEM
C     FIRST MCON ROWS COMPRISE MCON EQUALITY CONSTRAINTS
C
C *********************************************************************
C
C     TO BE USED IN CONNECTION WITH SUBROUTINE DECCON
C
C     RESEARCH CODE FOR GENERAL (M,N)-MATRICES     V 19.01.1984
C
C     INPUT PARAMETERS (* MARKS INOUT PARAMETERS)
C     -----------------------------------------------
C
C        A(M,N)      SEE OUTPUT OF DECCON
C        NROW        SEE OUTPUT OF DECCON
C        NCOL        SEE OUTPUT OF DECCON
C        M           SEE OUTPUT OF DECCON
C        N           SEE OUTPUT OF DECCON
C        MCON        SEE OUTPUT OF DECCON
C        IRANK       SEE OUTPUT OF DECCON
C        D(N)        SEE OUTPUT OF DECCON
C        PIVOT(N)    SEE OUTPUT OF DECCON
C        AH(N,N)     SEE OUTPUT OF DECCON
C        KRED        SEE OUTPUT OF DECCON
C      * B(M)        RIGHT-HAND SIDE OF LINEAR SYSTEM, IF (KRED.GE.0)
C                    RIGHT-HAND SIDE OF UPPER LINEAR SYSTEM,
C                                                      IF (KRED.LT.0)
C        V(N)        REAL WORK ARRAY
C
C     OUTPUT PARAMETERS
C     -----------------
C
C        X(N)        BEST LSQ-SOLUTION OF LINEAR SYSTEM
C        B(M)        RIGHT-HAND OF UPPER TRIGULAR SYSTEM
C                    (TRANSFORMED RIGHT-HAND SIDE OF LINEAR SYSTEM)
C
C
      INTEGER  I, II, I1, IH, IRK1, J, JJ, J1, MH
      INTEGER  IRANK, KRED, M, MCON, N, NROW, NCOL, PIVOT(N)
      DOUBLE PRECISION A(NROW,NCOL), AH(NCOL,NCOL)
      DOUBLE PRECISION B(M), D(N), V(N), X(N), S, ZERO
C
C     COMMON /MACHIN/ EPMACH, SMALL
C
C
      PARAMETER( ZERO=0.D0 )
C
C---1. SOLUTION FOR PSEUDO-RANK ZERO
C
      IF (IRANK.EQ.0)  THEN
         DO 1000 I=1,N
1000        X(I) = ZERO
C        ENDDO
         RETURN
      ENDIF
C
      IF (KRED.GE.0 .AND. (M.NE.1 .OR. N.NE.1) ) THEN
C
C---2. CONSTRAINED HOUSEHOLDER TRANSFORMATIONS OF RIGHT-HAND SIDE
C
         MH = MCON
         IF (MH.EQ.0)  MH = M
         DO  2100 J=1,IRANK
            S = ZERO
            DO  2110 I=J,MH
2110           S = S+A(I,J)*B(I)
C           ENDDO
            S = S/(D(J)*A(J,J))
            DO  2120 I=J,M
2120           B(I) = B(I)+A(I,J)*S
C           ENDDO
            IF (J.EQ.MCON)  MH = M
2100     CONTINUE
C        ENDDO
      ENDIF
C
C---3.1  SOLUTION OF UPPER TRIANGULAR SYSTEM
C
      IRK1 = IRANK+1
      DO  3100 II=1,IRANK
         I = IRK1-II
         I1 = I + 1
         S = B(I)
         IF (I1.LE.IRANK)  THEN
            DO  3111  JJ=I1,IRANK
3111           S = S-A(I,JJ)*V(JJ)
C           ENDDO
         ENDIF
3100     V(I) = S/D(I)
C     ENDDO
      IF (IRK1.LE.N) THEN
C
C---3.2  COMPUTATION OF THE BEST CONSTRAINED LSQ-SOLUTION
C
         DO  3210 J=IRK1,N
            S = ZERO
            J1 = J-1
            DO  3211  I=1,J1
3211           S = S+AH(I,J)*V(I)
C           ENDDO
3210        V(J) = -S/D(J)
C        ENDDO
         DO  3220 JJ=1,N
            J = N-JJ+1
            S = ZERO
            IF (JJ.NE.1) THEN
               DO  3221  I=J1,N
3221              S = S+AH(J,I)*V(I)
C              ENDDO
               IF (J.LE.IRANK) THEN
                  V(J) = V(J)-S
                  GOTO 3220
               ENDIF
            ENDIF
            J1=J
            V(J)=-(V(J)+S)/D(J)
3220     CONTINUE
C        ENDDO
      ENDIF
C
C---4. BACK-PERMUTATION OF SOLUTION COMPONENTS
C
      DO  4000 J=1,N
         IH=PIVOT(J)
4000     X(IH) = V(J)
C     ENDDO
      RETURN
C
C     **********  LAST CARD OF SOLCON  **********
C
      END
