#charset "us-ascii"
#include "adv3.h"

/*

Tads-3 Sense Connector Filter "Light" -- ALPHA

Steve Breslin, 2004
email: versim@hotmail.com

====

Licence:

Everyone is free to use this, but please alert me to your findings, and
provide me with all improvements and modifications.

====

Warning: the "Light" module is a slightly simplified version of the
full version of Sense Connector Filter. It is faster, but breaks with
the library slightly. If you have light passing through connectors to
illuminate otherwise darkened objects, that illumination will not work
under certain conditions. As a precaution, if parts of your game use
dimished light or darkness, you should use the full version of Sense
Connector Filter instead. (This only really matters if you're also
using Sense Connectors connecting to darkened locations.)

About this module:

This is a test module. Please profile any test cases both with and
without the module, and publicise your comparative findings.

Generally you'll need several sense connectors in relation to each
other in a string for this to be of use. If, for example, the player is
normally permitted to see objects in neighboring rooms, this module
will speed things up tremendously.

The purpose of this module is to decrease the size of the containment
table by stopping SenseConnectors from adding their locations to the
table in certain cases. (Basically, we filter when the SenseConnector's
locations cannot possibly be sensed by the POV.)

This is good because the smaller the containment table, the less work
later for the computation-heavy sense-path resolution.

Please note: this module will actually slow things down slightly if
it never succeeds to filter out any connections.

====

The algorithm is fairly simple:

Before SenseConnector.addDirectConnections() begins adding its
containers, it adds a transparency information object to the vector
libGlobal.transObjs. This object encapsulates the SenseConnector's
net transparency information for all senses. (Its net transparency in
the sense that it considers the transparency of that connector plus
the transparency of all the connectors backchain.)

When SenseConnector.addDirectConnections finishes adding its
containers, it removes its transparency object from the transObjs
vector.

Before SenseConnector.addDirectConnections() decides to add its
containers, it checks each of its non-opaque senses (from
transSensingThru()) against the transparency information in the
transObjs vector. If, on comparison, it fails to find at least one
non-opaque sense, it does not adds its containers.

The method SenseConnector.addDirectConnections(tab) is responsible for
adding the connector's locations to the table, so that's what we
modify.

====

Further clarification:

Where there's a bunch of sense-connectors connecting different rooms,
it's often the case that one connector adds to the containment table
another connector, with perhaps a third (and fourth and fifth, etc.)
further up the chain. If the transparency of any connector is canceled
by the net transparency of the connectors further up the chain, i.e.,
if the net transparency of each sense is 'opaque,' we don't let the
connector add its further locations (and further connectors).

The chain of sense connectors we're talking about is really the
call-chain we have during containment table construction. It's not to
be confused with sense-paths, which are calculated later.

====

Notes:

This module is largely untested. It is meant to work in all situations
or "room architectures" supported by the main library, but it's
entirely possible that you find a situation which this module doesn't
handle properly. Definitely please let me know!

For simplicity, this module only considers the basic senses defined by
the library: sight, smell, sound, and touch.

The library already relativizes the connection table to the POV. In the
library's version of the connection table, an object's presence in the
table does not mean that the table contains all objects possibly
sensible by this object. This is true with this module as well; we only
remark that our filtering mechanism does not stray from the library
model in this regard.

*/

modify libGlobal
    transObjs = static new Vector(3)
;

modify SenseConnector
    addDirectConnections(tab)
    {
        /* add myself */
        tab[self] = true;
            
        /* add my contents */
        foreach (local cur in contents)
        {
            if (tab[cur] == nil)
                cur.addDirectConnections(tab);
        }

//--!! This is where the modification comes.

        local myInfoObject;

        /* Create a new transInfoObject which encapsulates our
         * transparence information.
         *
         * At this point, the myInfoObject reflects our transparency of
         * each sense.
         */
        myInfoObject = new transObjInfo(self);

        /* If there is one or more sense-connector further up the call
         * chain, we calculate net the transparency of the last
         * (closest) one compared with the new one. (The last one
         * already encapsulates the net transparency of the entire
         * chain of connectors, just as the current one will by the end
         * of the following code block.)
         */
        if (libGlobal.transObjs.length())
        {
            local lastTrans = libGlobal.transObjs[libGlobal.transObjs.length];

            /* We iterate through the senses which are not already
             * opaque, and update the myInfoObject.
             */
            foreach(local curSense in lastTrans.nonOpaques)
                if (myInfoObject.nonOpaques.indexOf(curSense))
                    myInfoObject.updateSense(curSense, lastTrans);

            /* At this point, myInfoObject reflects the net
             * transparency of all connectors in the chain.
             */

            /* If there is no non-opaque sense passable by this connector,
             * we return without adding our locations.
             */
            if (myInfoObject.nonOpaques.length() == 0)
            {
                return;
            }
        }
 
       /* We're going to add our locations. The call to
         * cur.addDirectConnections(tab) can wind up calling yet
         * another sense connector. So we'll record our
         * transparency information in the libGlobal.transObjs
         * vector.
         */
        libGlobal.transObjs.append(myInfoObject);

        /* The following process is what the current library does,
         * without our mechanism.
         */
        foreach (local cur in locationList)
        {
            if (tab[cur] == nil)
            cur.addDirectConnections(tab);
        }

        /* Now that we've added all our locations (which have added
         * everything in them including any sense connectors,
         * recursively), we must remove our transObjInfo from the
         * transObjs vector.
         * That way, we won't interfere in the sense connectors
         * which aren't being connected to the POV by us.
         */
         libGlobal.transObjs.removeElementAt(libGlobal.transObjs.length());
    }
;

transObjInfo: object
    construct(connector)
    {
        nonOpaques = new Vector(4, allSenses);
        senseTable = new LookupTable();

        /* populate the senseTable and nonOpaques vector. */
        foreach(local curSense in allSenses)
            updateTable(curSense, connector.transSensingThru(curSense));
    }

    senseTable = nil
    nonOpaques = nil

    updateTable(sense, transparency)
    {
        /* We're not calculating attenuation. We'll consider this
         * transparency level to be equivalent to transparent for
         * our purposes.
         */
        if (transparency == attenuated)
            transparency = transparent;
        senseTable[sense] = transparency;
        if (transparency == opaque)
            nonOpaques.removeElement(sense);
    }

    updateSense(sense, obj)
    {
        local myTrans = senseTable[sense];
        local otherTrans = obj.senseTable[sense];

        /* two non-transparents equals opaque */
        if (myTrans != transparent && otherTrans != transparent)
                updateTable(sense, opaque);

        /* one non-transparent equals obscured (or distant); it doesn't
         * matter which we call it for our purposes.
         */
        else if (myTrans != transparent || otherTrans != transparent)
                updateTable(sense, obscured);
    }
    allSenses = [sight, smell, sound, touch]
;

